*---------------------------------------------------------------*
*	Assember options					*
*---------------------------------------------------------------*
	
	OPT		W-		; turn warnings off
	COMMENT		HEAD=7		; Fastload & TT flags

*---------------------------------------------------------------*
*	Header files						*
*---------------------------------------------------------------*

	include		stdmac.s
	include		romcfg.s
	include		pmmu.s

*---------------------------------------------------------------*
*	Program version						*
*---------------------------------------------------------------*

version			=	$507

*---------------------------------------------------------------*
*	Program cookie ID					*
*---------------------------------------------------------------*

ID			=	'TK40'

*---------------------------------------------------------------*
*	Stack sizes						*
*---------------------------------------------------------------*

user_stacksize		=	1024
super_stacksize		=	1024

*---------------------------------------------------------------*
*	Relocation base for patched ROM functions		*
*---------------------------------------------------------------*

rom_block_base		=	$00E64000

*---------------------------------------------------------------*
*	Relocation base for cold-reset function			*
*---------------------------------------------------------------*

coldreset_base		=	$00000600

*---------------------------------------------------------------*
*	System page definitions					*
*---------------------------------------------------------------*

syspage_s		=	2
syspage_e		=	5

*---------------------------------------------------------------*
*	Program specific structures				*
*---------------------------------------------------------------*

			rsreset
tkrd_cacr		rs.l	1
tkrd_cacrlock		rs.l	1
tkrd_slen		rs.b	0

*---------------------------------------------------------------*
*	External structures					*
*---------------------------------------------------------------*

mmu_slen		=	24
			rsset	-mmu_slen
mmu_tag			rs.l	1
mmu_id			rs.l	1
mmu_root		rs.l	1
mmu_end			rs.l	1
mmu_fastpages		rs.l	1
mmu_scrc		rs.l	1

xbra_slen		=	12
			rsset	-xbra_slen
xbra_tag		rs.l	1
xbra_id			rs.l	1
xbra_ptr		rs.l	1

*---------------------------------------------------------------*
*	Hardware-specific equates				*
*---------------------------------------------------------------*

pd_writethrough		=	%0000000000000
pd_copyback		=	%0000000100000
pd_serialized		=	%0000001000000
pd_nonserialized	=	%0000001100000
pd_writeprotect		=	%0000000000100
pd_present		=	%0000000000011
pd_supervisor		=	%0000010000000

ic4_bit			=	15
dc4_bit			=	31

dc4			=	(1<<dc4_bit)
ic4			=	(1<<ic4_bit)
dc4mask			=	~(dc4)
ic4mask			=	~(ic4)

*---------------------------------------------------------------*
*	Vectors & system variables				*
*---------------------------------------------------------------*

trap0			=	$80
trap1			=	$84
trap2			=	$88
trap13			=	$b4
trap14			=	$b8

reset_magic		=	$426
reset_vec		=	$42a

cookie_jar		=	$5a0

vbi_vec			=	$70

*---------------------------------------------------------------*
*	Program specific equates				*
*---------------------------------------------------------------*

bank_megs		=	32				; SIMM bank size
bank_count		=	2				; total number of SIMM banks

pmmu_mapsize		=	16+(bank_megs*bank_count)+16+32

jar_size		=	128				; new cookie jar size

xfrb_buffersize		=	64*1024				; XFRB disk buffer size

bank1			=	(16+(bank_megs*0))*(1024*1024)	; physical start of bank #1
bank2			=	(16+(bank_megs*1))*(1024*1024)	; physical start of bank #2
bank1b			=	(16+(bank_megs*2))*(1024*1024)	; physical start of bank #1b
bank2b			=	(16+(bank_megs*3))*(1024*1024)	; physical start of bank #2b

alt_newbase		=	16*1024*1024			; logical (final) address

max_blocks		=	64				; max possible SIMM fragments

*---------------------------------------------------------------*
*	General equates						*
*---------------------------------------------------------------*

			rsreset
STRAM_only		rs.b	1
VRAM_only		rs.b	1
STRAM_preferred		rs.b	1
VRAM_preferred		rs.b	1

none			=	-1

*---------------------------------------------------------------*
*	Program specific macros					*
*---------------------------------------------------------------*

check	macro
	tst.w		config_table+config_\1
	endm

*---------------------------------------------------------------*
*	Debugging macros					*
*---------------------------------------------------------------*

scol	macro
	move.l		#((\1<<24)|(\2<<16)|\3),$FFFF9800.w
	move.w		#(((\1>>4)<<8)|((\2>>4)<<4)|(\3>>4)),$FFFF8240.w
	endm

*---------------------------------------------------------------*
*	MMU macros						*
*---------------------------------------------------------------*
	
ttwrite	macro
	move.l		a1,d0
	and.l		#$FF000000,d0
	move.w		#$E040,d0
	movec		d0,dtt0
	move.w		\1,(a1)+
	endm

xtree	macro
	push.w		sr
	or.w		#$700,sr
	movec		dtt0,d4
	move.l		#$00FFE040,d3
	movec		d3,dtt0
	move.l		\1,\2
	movec		d4,dtt0
	pop.w		sr
	endm

*---------------------------------------------------------------*
*	Task specific constants					*
*---------------------------------------------------------------*

			rsreset
bp_lowtpa		rs.l	1
bp_hitpa		rs.l	1
bp_tbase		rs.l	1
bp_tlen			rs.l	1
bp_dbase		rs.l	1
bp_dlen			rs.l	1
bp_bbase		rs.l	1
bp_blen			rs.l	1
bp_dtaptr		rs.l	1
bp_parentbp		rs.l	1
bp_reserved1		rs.l	1
bp_environ		rs.l	1
bp_reserved2		rs.b	80
bp_cli_len		rs.b	1
bp_cli_string		rs.b	127
bp_slen			rs.b	0
	
*---------------------------------------------------------------*
*	False program start					*
*---------------------------------------------------------------*
start:	bra.l		real_start
*---------------------------------------------------------------*
*	Configuration table space				*
*---------------------------------------------------------------*
config_table:
*---------------------------------------------------------------*
xconfig_version_id	dc.l	config_id	; identification code
*---------------------------------------------------------------*
config_table_start:
*---------------------------------------------------------------*
xconfig_cpu_ic		dc.w	-1
xconfig_cpu_dc		dc.w	0	 
xconfig_cpu_pexec	dc.w	0
xconfig_sp_mode		dc.w	-1	; SystemPages Writetrough=$0000 Copyback=$ffff
xconfig_sp_type		dc.w	-1	; SystemPages ST-RAM=$0000 FastRAM=$ffff 
xconfig_sys_gemram	dc.w	-1
xconfig_sys_nemesis	dc.w	-1
xconfig_rom_logo	dc.w	0
xconfig_rom_wp		dc.w	0
xconfig_ram_alloc	dc.w	-1
xconfig_ram_bank2	dc.w	-1
xconfig_ram_check	dc.w	0
xconfig_d_mask		dc.w	-1	; 4MB=$0000 14MB=$ffff
xconfig_d_mode		dc.w	3	; Writetrough=$0000 Copyback=$0001 Serialized=$0002 Noneserialized=$0003
xconfig_d_novasupport	dc.w	0
xconfig_d_novablanker	dc.w	0
*---------------------------------------------------------------*
config_table_end:
*---------------------------------------------------------------*

msg_signon:		dc.b	13,10
			dc.b	27,'p'
			dc.b	' Afterburner 68040 ToolKit driver v'
			dc.b	'0'+((version/$100)&($F))
			dc.b	'.'
			dc.b	'0'+((version/$10)&($F))
			dc.b	'0'+((version)&($F))
			dc.b	'b ',13,10
;			dc.b	' Afterburner 68040 ToolKit driver v0.00b '
			dc.b	' Release date: <Sunday, 20th June 1997>  ',13,10
			dc.b	27,'q'
			dc.b	0
			even

*---------------------------------------------------------------*
scrc_end:		dc.l	0
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Real program start					*
*---------------------------------------------------------------*
real_start:
*---------------------------------------------------------------*
	move.l		4(sp),a5
	lea		usp_stack,sp
	move.l		a5,basepage
*---------------------------------------------------------------*
*	Mshrink - required for MMU malloc-shield		*
*---------------------------------------------------------------*
	move.l		#bp_slen,d0
	add.l		bp_tlen(a5),d0
	add.l		bp_dlen(a5),d0
	add.l		#(bss_end-bss_start)+(4-1),d0
	and.w		#-4,d0
	move.l		d0,pterm_length
	move.l		d0,-(sp)
	move.l		a5,-(sp)
	pea		$004a0000
	trap		#1
	lea		12(sp),sp
*---------------------------------------------------------------*
*	Enter supervisor mode					*
*---------------------------------------------------------------*
	move.l		sp,os_usp_stack
	pea		ssp_stack
	push.w		#32
	trap		#1
	move.l		d0,os_ssp_stack
*---------------------------------------------------------------*
*	Check for TK40						*
*---------------------------------------------------------------*
	move.l		#ID,d0
	bsr		cookie_search
	tst.l		d0
	sne		stay_resident		
*---------------------------------------------------------------*
*	Driver already installed - display error and exit	*
*---------------------------------------------------------------*
	tst.b		stay_resident
	bne		first_install
	lea		msg_already,a0
	bsr		cconws
	bra		super_exit
*---------------------------------------------------------------*
*	Driver not already installed				*
*---------------------------------------------------------------*
first_install:	
*---------------------------------------------------------------*
*	Sign-on message						*
*---------------------------------------------------------------*
	lea		msg_signon,a0
	bsr		cconws
*---------------------------------------------------------------*
*	Check for Magic						*
*---------------------------------------------------------------*
	move.l		#'MagX',d0
	bsr		cookie_search
	tst.l		d0
	seq		magic_resident
*---------------------------------------------------------------*
*	Install patches & other changes				*
*---------------------------------------------------------------*
	bsr		install_driver
*---------------------------------------------------------------*
*	Either exit program or warm-reset			*
*---------------------------------------------------------------*
exit_program:
*---------------------------------------------------------------*	
*	Ignore bootstrap if PMMU is resident			*
*---------------------------------------------------------------*	
	tst.b		pmmu_resident
	bne.s		skip_bootstrap
*---------------------------------------------------------------*
*	Error message if FastRAM missing on first pass		*
*---------------------------------------------------------------*
	tst.w		memory_blocks
	bne.s		.ccont
	lea		msg_nofastram,a0
	bsr		cconws
	bra		super_exit
*---------------------------------------------------------------*	
*	Ignore bootstrap if FastROM not available		*
*---------------------------------------------------------------*	
.ccont:	tst.b		fastrom_installed
	beq.s		skip_bootstrap
*---------------------------------------------------------------*	
*	Warm reboot						*
*---------------------------------------------------------------*	
	move.l		$4.w,a0
	jmp		(a0)
*---------------------------------------------------------------*
*	Bootstrap not required - finish & terminate		*
*---------------------------------------------------------------*
skip_bootstrap:
*---------------------------------------------------------------*
*	Sign-off message					*
*---------------------------------------------------------------*
.skip:	lea		msg_installed,a0
	bsr		cconws
*---------------------------------------------------------------*	
*	User mode						*
*---------------------------------------------------------------*	
super_exit
*---------------------------------------------------------------*	
	push.l		os_ssp_stack
	push.w		#32
	trap		#1
	move.l		os_usp_stack,sp
*---------------------------------------------------------------*
*	Terminate [ & stay resident ]				*
*---------------------------------------------------------------*
	push.w		#0
	tst.b		stay_resident
	beq.s		.pterm
	push.l		pterm_length
	push.w		#49
.pterm:	trap		#1

*---------------------------------------------------------------*
install_driver:
*---------------------------------------------------------------*
*	Calculate CRC for version and settings			*
*---------------------------------------------------------------*
	jsr		calculate_scrc
*---------------------------------------------------------------*
*	Reset MMU to minimum config, or to last tree (MagiC)	*
*---------------------------------------------------------------*
	bsr		reset_pmmu
*---------------------------------------------------------------*
*	Initialise processor caches to safe state		*
*---------------------------------------------------------------*
	bsr		init_caches
*---------------------------------------------------------------*
*	Install any FastRAM & build new MMU tree if needed	*
*---------------------------------------------------------------*
	bsr		install_fastram
*---------------------------------------------------------------*
*	Install Bios/XBios Trap & HD patches			*
*---------------------------------------------------------------*
	bsr		install_patches	
*---------------------------------------------------------------*
*	Mark low areas copyback					*
*---------------------------------------------------------------*
	bsr		setup_cbc
*---------------------------------------------------------------*
*	Set up CPU caches on termination			*
*---------------------------------------------------------------*
	bsr		setup_caches
*---------------------------------------------------------------*
*	Install/update new cookie jar + cookies			*
*---------------------------------------------------------------*
	bsr		cookie_manager
*---------------------------------------------------------------*
.skip	rts

*---------------------------------------------------------------*
cookie_manager:
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	beq.s		.skip_cookies
*---------------------------------------------------------------*
*	Install nice new cookie jar with LOTS of slots		*
*---------------------------------------------------------------*
	bsr		new_cookiejar
*---------------------------------------------------------------*
*	Upgrade CPU cookie to 68040				*
*---------------------------------------------------------------*
	jsr		upgrade_cpu
*---------------------------------------------------------------*
*	Upgrade FPU cookie to 68040				*
*---------------------------------------------------------------*
	jsr		upgrade_fpu
*---------------------------------------------------------------*
*	Install _FRB cookie & mark cookie buffers for DMA	*
*---------------------------------------------------------------*
	jsr		install_frb
*---------------------------------------------------------------*
*	Install ID, AB40 & XMMU cookies				*
*---------------------------------------------------------------*
	jsr		install_tk_cookies
*---------------------------------------------------------------*
*	Add special 'MOGR' cookie				*
*---------------------------------------------------------------*
	bsr		gemram_emulation
*---------------------------------------------------------------*
.skip_cookies:
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Create new cookie jar					*
*---------------------------------------------------------------*
new_cookiejar:
*---------------------------------------------------------------*
	move.l		cookie_jar.w,a0
	lea		new_jar,a1
	move.l		#jar_size,d3
	moveq		#0,d2
	move.l		a1,a2
.next:	move.l		(a0)+,d0
	beq.s		.done
	move.l		d0,(a2)+
	move.l		(a0)+,(a2)+
	addq.l		#1,d2
	bra.s		.next
.done:	sub.l		(a0)+,d3
	clr.l		(a2)+
	move.l		d3,(a2)+
	move.l		a1,cookie_jar.w
	rts

*---------------------------------------------------------------*
install_frb:
*---------------------------------------------------------------*
*	Avoid unwanted or pointless installation		*
*---------------------------------------------------------------*
	check		ram_alloc
	beq		.no_xfrb
*---------------------------------------------------------------*
*	Check for presence of old FRB cookie			*
*---------------------------------------------------------------*
	move.l		#'_FRB',d0
	bsr		cookie_search
	tst.l		d0
	beq.s		.no_current_frb
*---------------------------------------------------------------*
*	Create new _FRB cookie					*
*---------------------------------------------------------------*
	push.w		#STRAM_only
	push.l		#(xfrb_buffersize+8192)
	push.w		#68
	trap		#1
	addq.l		#8,sp
	add.l		#8192-1,d0
	and.l		#-8192,d0
	move.l		d0,d2
	move.l		#'_FRB',d0
	bsr		cookie_add
*---------------------------------------------------------------*
.no_current_frb:
*---------------------------------------------------------------*
*	Ensure _FRB buffer is not cacheable!			*
*---------------------------------------------------------------*
	move.l		#'_FRB',d0
	bsr		cookie_search
	tst.l		d0
	bne.s		.no_sfrb
	move.l		4(a0),a0
	move.l		a0,a1
	add.l		#xfrb_buffersize,a1
	bsr		change_2_nonserialized
*---------------------------------------------------------------*
.no_sfrb:
*---------------------------------------------------------------*
*	Ensure XFRB buffer is not cacheable!			*
*---------------------------------------------------------------*
	move.l		#'XFRB',d0
	bsr		cookie_search
	tst.l		d0
	bne.s		.no_xfrb
	move.l		06(a0),a0
	move.l		10(a0),a1
	add.l		a0,a1
	bsr		change_2_nonserialized
*---------------------------------------------------------------*
.no_xfrb:
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
install_tk_cookies:
*---------------------------------------------------------------*
*	Install ToolKit ID cookie				*
*---------------------------------------------------------------*
	move.l		#tkrd_struct,d2
	move.l		#ID,d0
	bsr		cookie_add
*---------------------------------------------------------------*
*	Install AB40 cookie					*
*---------------------------------------------------------------*
	move.l		#tkrd_struct,d2
	move.l		#'AB40',d0
	bsr		cookie_add
*---------------------------------------------------------------*
*	Add special 'XMMU' cookie for programmers		*
*---------------------------------------------------------------*
	lea		pmmu_struct,a0
	movec		srp,a1
	move.l		a1,pmmu_tree_root(a0)
	add.l		#(tree_end-root_tables),a1
	move.l		a1,pmmu_tree_end(a0)
	move.l		a0,d2
	move.l		#'XMMU',d0
	bsr		cookie_add
	rts
	
*---------------------------------------------------------------*
*	Install MOGR cookie					*
*---------------------------------------------------------------*
gemram_emulation:
*---------------------------------------------------------------*
*	Avoid unwanted or pointless installation		*
*---------------------------------------------------------------*
	check		sys_gemram
	beq.s		.nmogr
*---------------------------------------------------------------*
	move.l		#'MOGR',d0
	bsr		cookie_search
	tst.l		d0
	beq.s		.nmogr
*---------------------------------------------------------------*
*	Create cookie						*
*---------------------------------------------------------------*
	move.l		#usrs_space,d2
	move.l		#'MOGR',d0
	bsr		cookie_add
*---------------------------------------------------------------*
.nmogr:	rts

*---------------------------------------------------------------*
init_caches:
*---------------------------------------------------------------*
*	Initialise cache register & semaphore			*
*---------------------------------------------------------------*
	lea		tkrd_struct,a0
	clr.l		tkrd_cacrlock(a0)
	clr.l		tkrd_cacr(a0)
*---------------------------------------------------------------*
*	Get ready to initialise caches				*
*---------------------------------------------------------------*
	push.w		sr
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Invalidate caches if this is the first boot		*
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	bne.s		.noinv
	nop
	cinva		bc
	nop
*---------------------------------------------------------------*
*	Set up CACR and flush old data				*
*---------------------------------------------------------------*
.noinv:	lea		tkrd_struct,a0
	move.l		#ic4,d0
	move.l		d0,tkrd_cacr(a0)
	nop
	cpusha		bc
	nop
	movec		d0,cacr
*---------------------------------------------------------------*
	pop.w		sr
	rts

*---------------------------------------------------------------*
*	Upgrade _CPU cookie to 68040				*
*---------------------------------------------------------------*
upgrade_cpu:
*---------------------------------------------------------------*
	move.l		#40,d2
	move.l		#'_CPU',d0
	bsr		cookie_write
	rts

*---------------------------------------------------------------*
*	Upgrade _FPU cookie to '68040 internal'			*
*---------------------------------------------------------------*
upgrade_fpu:
*---------------------------------------------------------------*
	move.l		#%0000000000001000<<16,d2
	move.l		#'_FPU',d0
	bsr		cookie_write
	rts

*---------------------------------------------------------------*
*	Enable copyback mode according to configuration		*
*---------------------------------------------------------------*
setup_cbc:
*---------------------------------------------------------------*
*	Copyback already active if PMMU is resident		*
*---------------------------------------------------------------*
	tst.b		magic_resident
	bne		.ncbc
	tst.b		pmmu_resident
	bne.s		.ncbc
*---------------------------------------------------------------*
	check		sp_mode
	beq.s		.ncbc
*---------------------------------------------------------------*
;	lea		(8192*4).l,a0
;	lea		(8192*5).l,a1
;	lea		(8192*1).l,a0	; ST-RAM / low system areas.
;	lea		(8192*6).l,a1	; Do NOT mark the first page!
;	lea		syspage_s*8192,a0
;	lea		(syspage_s+syspage_e)*8192,a1
*---------------------------------------------------------------*
*	Mark pages as copyback					*
*---------------------------------------------------------------*
;	move.l		a0,a6
;	move.l		a1,d7
;	sub.l		a0,d7
;	lsr.l		#13-8,d7	
;	lsr.l		#8,d7	
;	moveq		#(pd_present|pd_copyback),d0
;	movec		srp,a1
;	bsr		edit_pages
*---------------------------------------------------------------*
*	Find last FastRAM page					*
*---------------------------------------------------------------*
	cmp.l		#$1357bd13,$5a8.w	; magic
	bne.s		.skip
	move.l		$5a4.w,a1
	move.l		a1,a0
	sub.l		#8192*8,a0
*---------------------------------------------------------------*
*	Mark pages as copyback					*
*---------------------------------------------------------------*
	jsr		change_2_copyback
*---------------------------------------------------------------*
*	Make sure PMMU tree is non-cacheable			*
*---------------------------------------------------------------*
.skip:	;bsr		mask_tree
.ncbc:	rts
	
*---------------------------------------------------------------*
*	Enable caches according to configuration		*
*---------------------------------------------------------------*
setup_caches:
*---------------------------------------------------------------*
	moveq		#0,d0
	lea		tkrd_struct,a0
*---------------------------------------------------------------*
	check		cpu_dc
	beq.s		.ndc
	bset		#dc4_bit,d0
*---------------------------------------------------------------*
.ndc:	check		cpu_ic
	beq.s		.nic
	bset		#ic4_bit,d0
*---------------------------------------------------------------*
.nic:	push.w		sr
	or.w		#$700,sr
	move.l		d0,tkrd_cacr(a0)
	nop
	cpusha		bc
	nop
	movec		d0,cacr
	pop.w		sr
	rts

*---------------------------------------------------------------*
calculate_scrc:
*---------------------------------------------------------------*
	clr.l		d2
*---------------------------------------------------------------*
	lea		config_table_start,a0
	move.l		#config_table_end,d0
	sub.l		a0,d0
	lsr.l		#1,d0
*---------------------------------------------------------------*
	moveq		#1,d1
*---------------------------------------------------------------*
.scrc:	tst.w		(a0)+
	beq.s		.skip
	add.l		d1,d2
.skip:	add.l		d1,d1
	subq.l		#1,d0
	bne.s		.scrc
*---------------------------------------------------------------*
	lea		start,a0
	move.l		#scrc_end,d0
	sub.l		a0,d0
	lsr.l		#2,d0
*---------------------------------------------------------------*
.pcrc:	move.l		(a0)+,d1
	eor.l		d1,d2
	subq.l		#1,d0
	bne.s		.pcrc
*---------------------------------------------------------------*
	move.l		d2,current_scrc
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
reset_pmmu:
*---------------------------------------------------------------*
	move.b		magic_resident,pmmu_resident
	bne		.err
*---------------------------------------------------------------*
*	Fetch current configuration CRC				*
*---------------------------------------------------------------*
	move.l		current_scrc,a1
*---------------------------------------------------------------*
*	Lock out interrupts					*
*---------------------------------------------------------------*
	move.w		sr,d7
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Record xTTx register state				*
*---------------------------------------------------------------*
	movec		dtt0,d1
*---------------------------------------------------------------*
*	Disable caches						*
*---------------------------------------------------------------*
	moveq		#0,d0
	movec		d0,cacr
*---------------------------------------------------------------*
*	Transparently translate all ram				*
*---------------------------------------------------------------*
	move.l		#$00FFE040,d0
	movec		d0,dtt0	
	nop
*---------------------------------------------------------------*
*	Flag for 'no tree present'				*
*---------------------------------------------------------------*
	moveq		#0,d6
	moveq		#0,d5
	moveq		#0,d4
*---------------------------------------------------------------*
*	PMMU is invalid if SRP is not in STRAM			*
*---------------------------------------------------------------*
	movec		tc,d0
	cmp.l		#$0000C000,d0
	bne		.no_tree
	movec		srp,d0
	tst.l		d0
	beq		.no_tree
	cmp.l		#$05000000,d0
	bhs		.no_tree
	cmp.l		#$00E00000,d0
	blo.s		.cont
	cmp.l		#$01000000,d0
	blo.s		.no_tree
.cont:	move.l		d0,a0
*---------------------------------------------------------------*
*	Ensure PMMU tree is actually real data			*
*---------------------------------------------------------------*
	cmp.l		#'TREE',mmu_tag(a0)
	bne		.no_tree
	cmp.l		#'KEEP',mmu_id(a0)
	bne.s		.no_tree
*---------------------------------------------------------------*
*	Cold reboot if old driver version detected		*
*---------------------------------------------------------------*
	cmp.l		mmu_scrc(a0),a1
	bne		.coldstart
*---------------------------------------------------------------*
*	Retrieve old installation data				*
*---------------------------------------------------------------*
	move.l		mmu_fastpages(a0),d4
*---------------------------------------------------------------*
*	Flag for 'tree present'					*
*---------------------------------------------------------------*
	moveq		#1,d6
	movec		srp,d5
*---------------------------------------------------------------*
.no_tree:
*---------------------------------------------------------------*
*	Restore ram state & interrupts				*
*---------------------------------------------------------------*
	movec		d1,dtt0
	nop
*---------------------------------------------------------------*
*	Store old installation data				*
*---------------------------------------------------------------*
	move.l		d4,old_fastram_pages
	move.l		d5,tree_root_ptr
*---------------------------------------------------------------*
*	If tree is valid, leave MMU alone			*
*---------------------------------------------------------------*
	tst.l		d6
	beq.s		.none
*---------------------------------------------------------------*
*	Restore PMMU						*
*---------------------------------------------------------------*
	movec		srp,d0
	movec		d0,urp
	move.l		#$0000C000,d0
	movec		d0,tc
	moveq		#0,d0
	movec		d0,itt0
	movec		d0,itt1
	movec		d0,dtt0
	movec		d0,dtt1
	nop
	pflusha
	nop
	st		pmmu_resident
	bra.s		.done
*---------------------------------------------------------------*
*	PMMU tree is not valid					*
*---------------------------------------------------------------*
.none:	nop
	pflusha
	nop
	move.l		#$00FFE000,d0
	movec		d0,itt0
	move.l		#$00FFE040,d0
	movec		d0,dtt0
	moveq		#0,d0
	movec		d0,itt1
	movec		d0,dtt1
	moveq		#0,d0
	movec		d0,tc
	movec		d0,srp
	movec		d0,urp
	nop
	pflusha
	nop
	sf		pmmu_resident
*---------------------------------------------------------------*
*	Restore interrupts					*
*---------------------------------------------------------------*
.done:	move.w		d7,sr
*---------------------------------------------------------------*
.err:	rts

*---------------------------------------------------------------*
.coldstart:
*---------------------------------------------------------------*
*	Restore ram state & reboot				*
*---------------------------------------------------------------*
	movec		d1,dtt0
	nop
	jmp		coldstart

*---------------------------------------------------------------*
*	Search jar for specific cookie				*
*---------------------------------------------------------------*
cookie_search:
*---------------------------------------------------------------*
	move.l		cookie_jar.w,a0
.loop:	cmp.l		(a0),d0
	beq.s		.find
	tst.l		(a0)
	beq.s		.fail
	addq.l		#8,a0
	bra.s		.loop
.find:	moveq		#0,d0
	bra.s		.end
.fail:	moveq		#-1,d0
.end:	rts

*---------------------------------------------------------------*
*	Update cookie value					*
*---------------------------------------------------------------*
cookie_write:
*---------------------------------------------------------------*
	bsr		cookie_search
	tst.l		d0
	bne.s		.fail
	move.l		d2,4(a0)
.fail:	rts

*---------------------------------------------------------------*
*	Add new cookie to jar					*
*---------------------------------------------------------------*
cookie_add:
*---------------------------------------------------------------*
	push.l		d0
	moveq		#0,d0
	bsr		cookie_search
	pop.l		d0
	lea		8(a0),a1
	move.l		(a0),(a1)+
	move.l		d0,(a0)+
	move.l		(a0),(a1)+
	move.l		d2,(a0)+
	rts

*---------------------------------------------------------------*
*	Really crap delay for error messages			*
*---------------------------------------------------------------*
delay:
*---------------------------------------------------------------*
	mulu.w		#50,d0
	move.w		d0,d7
.wt:	push.w		d7
	push.w		#37
	trap		#14
	addq.l		#2,sp
	pop.w		d7
	dbra		d7,.wt
	rts
	
*---------------------------------------------------------------*
*	Install all vector patches				*
*---------------------------------------------------------------*
install_patches:
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	beq.s		.skip
*---------------------------------------------------------------*
*	Disable blitter before it is disconnected		*
*---------------------------------------------------------------*
	push.w		#0
	push.w		#64
	trap		#14
	addq.l		#4,sp 
*---------------------------------------------------------------*
*	Install general patches					*
*---------------------------------------------------------------*
	bsr		patch_gemdos
	bsr		patch_xbios
	bsr		patch_bios
	bsr		patch_rv
*---------------------------------------------------------------*
*	Install copyback flush pass				*
*---------------------------------------------------------------*
	check		d_novasupport
	bne		.sc
	cmp.w		#1,config_table+config_d_mode
	bne.s		.sc
	move.l		vbi_vec.w,old_copyback_flush
	move.l		#copyback_flush,vbi_vec.w
*---------------------------------------------------------------*
*	Install Nova video blanker				*
*---------------------------------------------------------------*
.sc:	check		d_novasupport
	beq		.skip
	check		d_novablanker
	beq		.skip
	move.l		vbi_vec.w,old_nova_blanker
	move.l		#nova_blanker,vbi_vec.w
*---------------------------------------------------------------*
.skip:	rts

*---------------------------------------------------------------*
patch_gemdos:
*---------------------------------------------------------------*
	move.l		trap1.w,old_trap1_rom
	move.l		#new_trap1_rom,trap1.w
	rts

*---------------------------------------------------------------*
patch_bios:
*---------------------------------------------------------------*
	check		sys_gemram
	beq.s		.no
	move.l		trap13.w,old_trap13
	move.l		#new_trap13,trap13.w
.no:	rts

*---------------------------------------------------------------*
patch_xbios:
*---------------------------------------------------------------*
	move.l		trap14.w,old_trap14
	move.l		#new_trap14,trap14.w
	rts
	
*---------------------------------------------------------------*
patch_rv:
*---------------------------------------------------------------*
;	move.l		reset_vec.w,old_rv
;	move.l		#new_rv,reset_vec.w
;	move.l		reset_magic.w,old_rm
;	move.l		#$31415926,reset_magic.w
	rts

*---------------------------------------------------------------*
old_rm:	ds.l		1	
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_rv:	ds.l		1
*---------------------------------------------------------------*
new_rv:
*---------------------------------------------------------------*
*	Lock out interrupts & make all ram transparent		*
*---------------------------------------------------------------*
	lea		nemcode_off,a1
	check		sys_nemesis
	beq.s		.nonem
	lea		nemcode_on,a1
.nonem:	lea		.rts,a0
	jmp		(a1)
*---------------------------------------------------------------*
.rts:	move.w		sr,d7
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Flush caches & ATC					*
*---------------------------------------------------------------*
	moveq		#0,d0
	movec		d0,cacr
	nop
	cinva		bc
	nop
	pflusha
	nop
*---------------------------------------------------------------*
	move.l		#$00FFE040,d0
	movec		d0,dtt0
	moveq		#0,d0
	movec		d0,dtt1
*---------------------------------------------------------------*
*	Flag for 'no tree present'				*
*---------------------------------------------------------------*
	moveq		#0,d6
*---------------------------------------------------------------*
*	PMMU is invalid if SRP is not in STRAM/FastRAM		*
*---------------------------------------------------------------*
	movec		srp,d0
	moveq		#2-1,d1
.lp:	tst.l		d0
	beq		.err
	cmp.l		#$05000000,d0
	bhs		.err
	cmp.l		#$00E00000,d0
	blo.s		.cont
	cmp.l		#$01000000,d0
	blo.s		.err
*---------------------------------------------------------------*
*	Ensure PMMU tree is actually real data			*
*---------------------------------------------------------------*
.cont:	move.l		d0,a0
	cmp.l		#'TREE',mmu_tag(a0)
	bne.s		.err
	cmp.l		#'KEEP',mmu_id(a0)
	bne.s		.err
	bra.s		.tree_ok
.err:

;	move.l		resident_ptr,d0
;	dbra		d1,.lp

	bra		.no_tree

*---------------------------------------------------------------*
.tree_ok:
*---------------------------------------------------------------*
	move.l		a0,a1
*---------------------------------------------------------------*
*	Flag for 'tree present'					*
*---------------------------------------------------------------*
	moveq		#1,d6
*---------------------------------------------------------------*
.no_tree:
*---------------------------------------------------------------*
*	Restore ram state & interrupts				*
*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	If tree is valid, leave MMU alone			*
*---------------------------------------------------------------*
	tst.l		d6
	beq.s		.none
*---------------------------------------------------------------*
*	Restore PMMU						*
*---------------------------------------------------------------*
	move.l		a1,d0
	movec		d0,srp
	movec		d0,urp
	move.l		#$0000C000,d0
	movec		d0,tc
	nop
	moveq		#0,d0
	movec		d0,itt0
	movec		d0,itt1
	movec		d0,dtt0
	movec		d0,dtt1
	nop
	pflusha
	nop
	bra.s		.done
*---------------------------------------------------------------*
*	PMMU tree is not valid					*
*---------------------------------------------------------------*
.none:	move.l		#$00FFE000,d0
	movec		d0,itt0
	move.l		#$00FFE040,d0
	movec		d0,dtt0
	moveq		#0,d0
	movec		d0,itt1
	movec		d0,dtt1
	nop
	moveq		#0,d0
	movec		d0,tc
	movec		d0,srp
	movec		d0,urp
	nop
	pflusha
	nop
*---------------------------------------------------------------*
.done:	move.l		#ic4,d0
	movec		d0,cacr
*---------------------------------------------------------------*
	move.w		d7,sr
*---------------------------------------------------------------*
*	Continue with next RV routine				*
*---------------------------------------------------------------*
	move.l		old_rv,reset_vec.w
	move.l		old_rm,reset_magic.w
	jmp		(a6)


*---------------------------------------------------------------*
	cnop		0,4
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_trap1:
*---------------------------------------------------------------*
	dc.l		0
*---------------------------------------------------------------*
new_trap1:
*---------------------------------------------------------------*
*	Locate caller stackframe & fetch function ID		*
*---------------------------------------------------------------*
	move.l		usp,a0
	btst		#5,(sp)
	beq.s		.user
	lea		8(sp),a0
.user:	move.w		(a0),d0
*---------------------------------------------------------------*
*	Intercept Pexec						*		
*---------------------------------------------------------------*
	cmp.w		#$4B,d0
	beq.s		new_trap1_pexec
*---------------------------------------------------------------*
*	Keep CACR up-to-date					*
*---------------------------------------------------------------*
	lea		tkrd_struct,a0
*---------------------------------------------------------------*
*	Abort CACR update if cache is locked			*
*---------------------------------------------------------------*
	tst.b		tkrd_cacrlock(a0)
	bne.s		continue1
*---------------------------------------------------------------*
*	Abort CACR update if CACR reflects correct values	*
*---------------------------------------------------------------*
	movec		cacr,d0
	cmp.l		tkrd_cacr(a0),d0
	beq.s		continue1
*---------------------------------------------------------------*
*	Flush caches and update CACR				*
*---------------------------------------------------------------*
	move.w		sr,d1
	or.w		#$700,sr
	nop
	cpusha		bc
	nop
	move.l		tkrd_cacr(a0),d0
	movec		d0,cacr
	movec		cacr,d0
	move.l		d0,tkrd_cacr(a0)
	move.w		d1,sr
*---------------------------------------------------------------*
continue1:
*---------------------------------------------------------------*
	move.l		old_trap1,a0
	jmp		(a0)
*---------------------------------------------------------------*
complete1:
*---------------------------------------------------------------*
	rte

*---------------------------------------------------------------*
new_trap1_pexec:
*---------------------------------------------------------------*
*	Disable instruction cache & continue			*
*---------------------------------------------------------------*
	movec		cacr,d0
	bclr		#ic4_bit,d0
	beq.s		continue1
*---------------------------------------------------------------*
*	If cache was previously enabled, then flush it		*
*---------------------------------------------------------------*
	movec		d0,cacr
	nop
	cinva		ic
	nop
	bra		continue1

*---------------------------------------------------------------*
	cnop		0,4
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_trap1_rom:
*---------------------------------------------------------------*
	dc.l		0
*---------------------------------------------------------------*
new_trap1_rom:
*---------------------------------------------------------------*
*	Keep CACR up-to-date					*
*---------------------------------------------------------------*
	lea		tkrd_struct,a0
*---------------------------------------------------------------*
*	Abort CACR update if cache is locked			*
*---------------------------------------------------------------*
	tst.b		tkrd_cacrlock(a0)
	bne.s		continue1_rom
*---------------------------------------------------------------*
*	Abort CACR update if CACR reflects correct values	*
*---------------------------------------------------------------*
	movec		cacr,d0
	cmp.l		tkrd_cacr(a0),d0
	beq.s		continue1_rom
*---------------------------------------------------------------*
*	Flush caches and update CACR				*
*---------------------------------------------------------------*
	move.w		sr,d1
	or.w		#$700,sr
	nop
	cpusha		bc
	nop
	move.l		tkrd_cacr(a0),d0
	movec		d0,cacr
	movec		cacr,d0
	move.l		d0,tkrd_cacr(a0)
	move.w		d1,sr
*---------------------------------------------------------------*
continue1_rom:
*---------------------------------------------------------------*
	move.l		old_trap1_rom,a0
	jmp		(a0)

*---------------------------------------------------------------*
	cnop		0,4
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_trap14:
*---------------------------------------------------------------*
	dc.l		0
*---------------------------------------------------------------*
new_trap14:
*---------------------------------------------------------------*
*	Locate caller stackframe & fetch function ID		*
*---------------------------------------------------------------*
	move.l		usp,a0
	btst		#5,(sp)
	beq.s		.user
	lea		8(sp),a0
.user:	move.w		(a0),d0
*---------------------------------------------------------------*
*	Intercept supexec if CBC is enabled			*
*---------------------------------------------------------------*
;	cmp.w		#38,d0
;	blo.s		continue14
;	beq.s		new_trap14_supexec
	cmp.w		#64,d0
	beq.s		new_trap14_blitmode
*---------------------------------------------------------------*
continue14:
*---------------------------------------------------------------*
	move.l		old_trap14,a0
	jmp		(a0)
*---------------------------------------------------------------*
complete14:
*---------------------------------------------------------------*
	rte

*---------------------------------------------------------------*
*	Flush datacache before supervisor calls			*
*---------------------------------------------------------------*
new_trap14_supexec:
*---------------------------------------------------------------*
*	Abort flush if CBC is not enabled			*
*---------------------------------------------------------------*
	check		sp_mode
	beq.s		continue14
*---------------------------------------------------------------*
*	Flush DC and continue					*
*---------------------------------------------------------------*
	nop
	cpusha		dc
	nop
	bra.s		continue14

*---------------------------------------------------------------*
*	Cut Blitter off from TOS				*
*---------------------------------------------------------------*
new_trap14_blitmode:	
*---------------------------------------------------------------*
*	Always return 'blitter not present & off'		*
*---------------------------------------------------------------*
	moveq		#0,d0
	bra.s		complete14

*---------------------------------------------------------------*
*	Install FastRAM & build new PMMU tree			*
*---------------------------------------------------------------*
install_fastram:
*---------------------------------------------------------------*
*	Skip installation if Magic is resident			*
*---------------------------------------------------------------*
	tst.b		magic_resident
	bne		.skip_install
*---------------------------------------------------------------*
*	Set up FastRAM base address				*
*---------------------------------------------------------------*
	move.l		#alt_newbase,alt_start
*---------------------------------------------------------------*
*	If MMU is reset-resident, then use last FastRAM cfg	*
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	beq.s		.build_new_tree
	move.l		old_fastram_pages,fastram_pages
	bra		.allocate_fastram
*---------------------------------------------------------------*
*	Search SIMM banks for ram fragments			*
*---------------------------------------------------------------*
.build_new_tree:
*---------------------------------------------------------------*
	lea		msg_searching,a0
	bsr		cconws
	bsr		memory_check
*---------------------------------------------------------------*
*	Abort if insufficient FastRAM is available		*
*---------------------------------------------------------------*
	tst.w		memory_blocks
	beq		.skip_install
*---------------------------------------------------------------*
*	Copy ROM image into FastRAM				*
*---------------------------------------------------------------*
	bsr		transfer_rom	
*---------------------------------------------------------------*
*	Make space for PMMU tree in FastRAM			*
*---------------------------------------------------------------*
	bsr		find_pmmu_start	
*---------------------------------------------------------------*
*	Make space for system page in FastRAM			*
*---------------------------------------------------------------*
	bsr		find_sys_start
*---------------------------------------------------------------*
*	Build new PMMU tree					*
*---------------------------------------------------------------*
	bsr		build_tree
*---------------------------------------------------------------*
*	Install FastROM patches					*
*---------------------------------------------------------------*
	bsr		patch_fastrom
*---------------------------------------------------------------*
*	Make TOS aware of new alternate memory			*
*---------------------------------------------------------------*
	tst.l		fastram_pages
	beq		.err
*---------------------------------------------------------------*
.allocate_fastram:
*---------------------------------------------------------------*
	move.l		fastram_pages,d0
	moveq		#13,d1
	lsl.l		d1,d0	
	push.l		d0
	push.l		d0
	add.l		alt_start,d0
	move.l		d0,alt_end
	push.l		alt_start
	move.w		#20,-(sp)		; Maddalt()
	check		ram_alloc
	beq.s		.skip1
	trap		#1
.skip1:	lea		10(sp),sp	
	pop.l		d0
*---------------------------------------------------------------*
*	Configure ALT ramtop & magic				*
*---------------------------------------------------------------*
	add.l		alt_start,d0
	check		ram_alloc
	beq.s		.skip2
	move.l		d0,$5a4.w		; end of alternate ram
	move.l		#$1357bd13,$5a8.w	; magic
*---------------------------------------------------------------*
*	Memory check						*
*---------------------------------------------------------------*
.skip2:	bsr		detect_faults
*---------------------------------------------------------------*
*	Memory controller configuration is valid		*
*---------------------------------------------------------------*
.err:	;move.l		#$752019f3,$420.w
	;move.l		#$237698aa,$43a.w
*---------------------------------------------------------------*
.skip_install:
*---------------------------------------------------------------*
	rts
	
*---------------------------------------------------------------*
build_tree:
*---------------------------------------------------------------*
*	Set up PMMU tree pointers				*
*---------------------------------------------------------------*
	bsr		init_table_pointers
*---------------------------------------------------------------*
*	Logical address space starts at $00000000		*
*---------------------------------------------------------------*
	lea		$00000000.w,a6
*---------------------------------------------------------------*
*	Create pages for first 16MB of space			*
*---------------------------------------------------------------*
	bsr		mark_compatible_16mb
*---------------------------------------------------------------*
*	Mark [n]MB of FastRAM (r/w/u/s/c)			*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,fastram_pageflags
	clr.l		fastram_pages
	lea		block_list,a5
	move.w		memory_blocks,d6
	bra.s		.ls
.log:	push.w		d6
	move.l		(a5)+,a4
	move.l		(a5)+,d7
	sub.l		a4,d7
	ble.s		.skip
	push.l		a5
	moveq		#13,d1
	lsr.l		d1,d7
	add.l		d7,fastram_pages
	move.l		fastram_pageflags,d0
	bsr		mark_8k
.skip:	pop.l		a5
	pop.w		d6
.ls:	dbra		d6,.log
*---------------------------------------------------------------*
*	Mark 32MB remainder as invalid space (u)		*
*---------------------------------------------------------------*
	move.l		a6,d7
	moveq		#13,d0
	lsr.l		d0,d7
	and.l		#((32*1024*1024)/8192)-1,d7
	beq.s		.none_remaining
	neg.l		d7
	add.l		#((32*1024*1024)/8192),d7
	moveq		#%0000000000000,d0
	bsr		mark_8k
*---------------------------------------------------------------*
.none_remaining:
*---------------------------------------------------------------*
	move.l		a6,shadow_base
*---------------------------------------------------------------*
*	Generate shadow for first 16MB				*
*---------------------------------------------------------------*
	bsr		mark_compatible_16mb
	bsr		mark_compatible_16mb
*---------------------------------------------------------------*
*	Find number of remaining 32MB blocks			*
*---------------------------------------------------------------*
	moveq		#32-7,d1
	move.l		a6,d7
	lsr.l		d1,d7
	move.l		#1<<7,d0
	sub.l		d7,d0
*---------------------------------------------------------------*
*	Mark 4GB remainder (minus 32MB) as mirrors of 1st 16MB	*
*---------------------------------------------------------------*
	move.l		shadow_base,d2
	lsr.l		d1,d2
	move.l		(a1,d2.l*4),d1
	lea		(a1,d7.l*4),a5
	bra.s		.go
.all	move.l		d1,(a5)+
.go:	dbra		d0,.all
*---------------------------------------------------------------*
*	Flag MMU pages as non-cacheable RAM			*
*---------------------------------------------------------------*
	bsr		mask_tree
*---------------------------------------------------------------*
*	Set up root pointer					*
*---------------------------------------------------------------*
	move.l		tree_root_ptr,a0
*---------------------------------------------------------------*
*	Mark PMMU tree header for identification		*
*---------------------------------------------------------------*
	move.l		#'TREE',mmu_tag(a0)
	move.l		#'KEEP',mmu_id(a0)
*---------------------------------------------------------------*
*	Record PMMU tree details in header			*
*---------------------------------------------------------------*
	move.l		tree_root_ptr,mmu_root(a0)
	move.l		tree_end_ptr,mmu_end(a0)
*---------------------------------------------------------------*
*	Record FastRAM details in header			*
*---------------------------------------------------------------*
	move.l		fastram_pages,mmu_fastpages(a0)
*---------------------------------------------------------------*
*	Record CRC in header					*
*---------------------------------------------------------------*
	move.l		current_scrc,mmu_scrc(a0)
*---------------------------------------------------------------*
*	Freeze exceptions					*
*---------------------------------------------------------------*
	push.w		sr
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Flush both CPU caches					*
*---------------------------------------------------------------*
	nop
	cpusha		bc
*---------------------------------------------------------------*
*	Flush AT cache						*
*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	Set up user & super root pointers			*
*---------------------------------------------------------------*
	movec		a0,urp
	movec		a0,srp
*---------------------------------------------------------------*
*	Enable PMMU translation tree				*
*---------------------------------------------------------------*
	move.l		#$0000C000,d0
	movec		d0,tc
*---------------------------------------------------------------*
*	Disable all transparent translation			*
*---------------------------------------------------------------*
	moveq		#0,d0
	movec		d0,dtt0
	movec		d0,dtt1
	movec		d0,itt0
	movec		d0,itt1
*---------------------------------------------------------------*
*	Flush AT cache						*
*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	Transfer system pages into fastram			*
*---------------------------------------------------------------*
	move.l		sys_start,d0
	beq.s		.err
	movec		dtt0,d2
	move.l		#$00FFE040,d1
	movec		d1,dtt0
	nop
	lea		(syspage_s*8192),a0
	move.l		d0,a1
	move.l		#((syspage_e-syspage_s)*8192)/4,d0
.copy:	move.l		(a0)+,(a1)+
	subq.l		#1,d0
	bne.s		.copy
	movec		d2,dtt0
	nop
*---------------------------------------------------------------*
*	Restore exceptions					*
*---------------------------------------------------------------*
.err:	pop.w		sr
	rts
	
*---------------------------------------------------------------*
*	Copy ROM to FastRAM					*
*---------------------------------------------------------------*
transfer_rom:
*---------------------------------------------------------------*
*	Skip pointless or unwanted installation			*
*---------------------------------------------------------------*
	tst.w		memory_blocks
	beq		.no_fastrom
*---------------------------------------------------------------*
*	Prevent ROM from being re-transferred			*
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	bne		.no_fastrom
*---------------------------------------------------------------*
	lea		msg_transferring,a0
	bsr		cconws
*---------------------------------------------------------------*
*	FastROM is valid					*
*---------------------------------------------------------------*
	st		fastrom_installed
*---------------------------------------------------------------*
*	Calculate ROM size					*
*---------------------------------------------------------------*
;	lea		$00E00000.l,a0
;	move.l		20(a0),d7
;	sub.l		a0,d7
;	add.l		#8192-1,d7
;	and.l		#-8192,d7
	move.l		#(512*1024),d7
	move.l		d7,rom_size
*---------------------------------------------------------------*
*	Take ROM from first available block			*
*---------------------------------------------------------------*
	bsr		find_rom_start
*---------------------------------------------------------------*
*	Calculate size of unused rom portion			*
*---------------------------------------------------------------*
	lea		$00E00000.l,a0
	move.l		rom_size,d7
	move.l		d7,d4
	sub.l		20(a0),d4
	move.l		rom_start,a1
*---------------------------------------------------------------*
*	Only decode used portion of ROM				*
*---------------------------------------------------------------*
.next:	cmp.l		d4,d7
	ble		.no_pmove
*---------------------------------------------------------------*
*	Fetch next opcode					*
*---------------------------------------------------------------*
	move.l		(a0),d1
*---------------------------------------------------------------*
*	Decode MOVEC->CACR opcodes				*
*---------------------------------------------------------------*
	move.l		d1,d0
	and.l		#$FFFF0FFF,d0
	cmp.l		#$4E7B0002,d0
	bne.s		.no_cacr
*---------------------------------------------------------------*
*	Open FastRAM for writing & output 2 nops		*
*---------------------------------------------------------------*
	move.w		sr,d2
	or.w		#$700,sr
	movec		dtt0,d3
	ttwrite		#$4E71
	ttwrite		#$4E71
	movec		d3,dtt0
	move.w		d2,sr
*---------------------------------------------------------------*
*	Advance source & loop count				*
*---------------------------------------------------------------*
	addq.l		#4,a0
	subq.l		#4,d7
	bgt.s		.next
	bra		.done
*---------------------------------------------------------------*
.no_cacr:
*---------------------------------------------------------------*
*	Decode PMOVE->TTx opcodes				*
*---------------------------------------------------------------*
	move.l		d1,d0
	swap		d0
	cmp.w		#$F039,d0
	bne.s		.no_pmove
	swap		d0
	and.w		#$B3FF,d0
	bne.s		.no_pmove
*---------------------------------------------------------------*
*	Open FastRAM for writing & output 4 nops		*
*---------------------------------------------------------------*
	move.w		sr,d2
	or.w		#$700,sr
	movec		dtt0,d3
	ttwrite		#$4E71
	ttwrite		#$4E71
	ttwrite		#$4E71
	ttwrite		#$4E71
	movec		d3,dtt0
	move.w		d2,sr
*---------------------------------------------------------------*
*	Advance source & loop count				*
*---------------------------------------------------------------*
	addq.l		#8,a0
	subq.l		#8,d7
	bgt		.next
	bra.s		.done
*---------------------------------------------------------------*
.no_pmove:
*---------------------------------------------------------------*
*	Normal copy						*
*---------------------------------------------------------------*
	move.w		sr,d2
	or.w		#$700,sr
	movec		dtt0,d3
	ttwrite		(a0)+
	movec		d3,dtt0
	move.w		d2,sr
	subq.l		#2,d7
	bgt		.next
*---------------------------------------------------------------*
*	Restore caches						*
*---------------------------------------------------------------*
.done:
*---------------------------------------------------------------*
.no_fastrom:
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Take ROM from first available block			*
*---------------------------------------------------------------*
find_rom_start:
*---------------------------------------------------------------*
	move.l		rom_size,d4
	lea		block_list,a5
	move.w		memory_blocks,d0
	bra.s		.sgo
.all:	move.l		0(a5),d1
	move.l		4(a5),d2
	move.l		d2,d3
	sub.l		d1,d3
	cmp.l		d4,d3
	bmi.s		.next
	move.l		d1,rom_start
	add.l		d4,d1
	move.l		d1,(a5)
	bra.s		.done
.next:	addq.l		#8,a5
.sgo:	dbra		d0,.all
.done:	rts

*---------------------------------------------------------------*
*	Take page from first available block			*
*---------------------------------------------------------------*
find_sys_start:
*---------------------------------------------------------------*
	clr.l		sys_start
	check		sp_type
	beq.s		.done
	move.l		#(syspage_e-syspage_s)*8192,d4
	lea		block_list,a5
	move.w		memory_blocks,d0
	bra.s		.sgo
.all:	move.l		0(a5),d1
	move.l		4(a5),d2
	move.l		d2,d3
	sub.l		d1,d3
	cmp.l		d4,d3
	bmi.s		.next
	move.l		d1,sys_start
	add.l		d4,d1
	move.l		d1,(a5)
	bra.s		.done
.next:	addq.l		#8,a5
.sgo:	dbra		d0,.all
.done:	rts

*---------------------------------------------------------------*
*	Take PMMU tree from first available block		*
*---------------------------------------------------------------*
find_pmmu_start:
*---------------------------------------------------------------*
*	Calculate PMMU tree size				*
*---------------------------------------------------------------*
	move.l		#(tree_end-root_tables)+8192+(8192-1),d4
	and.l		#-8192,d4
*---------------------------------------------------------------*
*	Set up default tree addresses				*
*---------------------------------------------------------------*
	move.l		#root_tables,d0
	and.l		#-512,d0
	move.l		d0,tree_root_ptr
*---------------------------------------------------------------*
*	Seach for suitable block				*
*---------------------------------------------------------------*
	lea		block_list,a5
	move.w		memory_blocks,d0
	bra.s		.sgo
.all:	move.l		0(a5),d1
	move.l		4(a5),d2
	move.l		d2,d3
	sub.l		d1,d3
	cmp.l		d4,d3
	bmi.s		.next
	move.l		d1,d7
	add.l		d4,d1
	move.l		d1,(a5)
	add.l		#8192,d7
	move.l		d7,tree_root_ptr
	bra.s		.done
.next:	addq.l		#8,a5
.sgo:	dbra		d0,.all
.done:	rts

*---------------------------------------------------------------*
mark_compatible_16mb:
*---------------------------------------------------------------*
*	Start of physical user RAM area = $00000000		*
*---------------------------------------------------------------*
	lea		$00000000.l,a4
*---------------------------------------------------------------*
*	Mark first pages as normal STRAM			*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,d0
	move.l		#syspage_s,d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mark system pages as (optional) FastRAM/CBC		*
*---------------------------------------------------------------*
	push.l		a4
	move.l		sys_start,d0
	beq.s		.ok
	move.l		d0,a4
.ok:	move.l		#pd_present|pd_writethrough,d0
	check		sp_mode
	beq.s		.norm
	move.l		#pd_present|pd_copyback,d0
.norm:	move.l		#(syspage_e-syspage_s),d7
	bsr		mark_8k
	pop.l		a4
*---------------------------------------------------------------*
*	Advance physical address past system pages		*
*---------------------------------------------------------------*
	add.l		#(syspage_e-syspage_s)*8192,a4
*---------------------------------------------------------------*
*	Mark 14MB STRAM (r/w/u/s/c)				*
*---------------------------------------------------------------*
	check		d_novasupport
	beq.s		.not_nova
*---------------------------------------------------------------*
*	Mark 10MB STRAM (r/w/u/s/c) (Nova)			*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,d0
	move.l		#(10*128)-syspage_e,d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mark 1MB mono display (Nova)				*
*---------------------------------------------------------------*
	bsr		get_displaymode
	move.l		#(1*128),d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mark 1MB H/W registers (Nova)				*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_serialized,d0
	move.l		#(1*128),d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mark 2MB colour display (Nova)				*
*---------------------------------------------------------------*
	bsr		get_displaymode
	move.l		#(2*128),d7
	bsr		mark_8k
*---------------------------------------------------------------*
	bra		.mdon
*---------------------------------------------------------------*
.not_nova:
*---------------------------------------------------------------*
	check		d_mask
	bne.s		.not_4meg
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,d0
	move.l		#(03*128)-syspage_e,d7
	bsr		mark_8k
	bsr		get_displaymode
	move.l		#(11*128),d7
	bsr		mark_8k
	bra		.mdon
*---------------------------------------------------------------*
.not_4meg:
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,d0
	move.l		#(13*128)-syspage_e,d7
	bsr		mark_8k
	bsr		get_displaymode
	move.l		#(01*128),d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mirror 1MB of ROM into FastRAM (r/w/u/s/c)		*
*---------------------------------------------------------------*
.mdon:	tst.b		fastrom_installed
	beq.s		.nfr
*---------------------------------------------------------------*
*	Mark FastROM pages as cacheable/noncacheable		*
*---------------------------------------------------------------*
	move.l		rom_start,a4
	move.l		rom_size,d7
	moveq		#13,d0
	lsr.l		d0,d7
	move.l		#pd_present|pd_writethrough,d0
	bsr		mark_8k
*---------------------------------------------------------------*
*	Mark remaining (unused) pages as normal ROM		*
*---------------------------------------------------------------*
	lea		$00E00000.l,a4
	move.l		rom_size,d7
	add.l		d7,a4
	neg.l		d7
	add.l		#(1*1024*1024),d7
	beq.s		.yfr
	moveq		#13,d0
	lsr.l		d0,d7
	move.l		#pd_present|pd_writethrough,d0
	bsr		mark_8k
	bra.s		.yfr
*---------------------------------------------------------------*
*	Start of physical ROM area = $00E00000			*
*---------------------------------------------------------------*
.nfr:	lea		$00E00000.l,a4
*---------------------------------------------------------------*
*	Mark hardware ROM pages as normal ROM			*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_writethrough,d0
	move.l		#(1*128),d7
	bsr		mark_8k
*---------------------------------------------------------------*
*	Start of physical hardware I/O space = $00F00000	*
*---------------------------------------------------------------*
.yfr:	lea		$00F00000.l,a4
*---------------------------------------------------------------*
*	Mark 1MB of H/W regs (r/w/s)				*
*---------------------------------------------------------------*
	move.l		#pd_present|pd_serialized,d0
	move.l		#(1*128),d7
	bsr		mark_8k
	rts

get_displaymode:
	move.w		config_table+config_d_mode,d7
	beq.s		.wt4
	subq.w		#1,d7
	beq.s		.cb4
	subq.w		#1,d7
	beq.s		.sr4
	move.l		#pd_present|pd_nonserialized,d0
	bra.s		.go4
.sr4:	move.l		#pd_present|pd_serialized,d0
	bra.s		.go4
.cb4:	move.l		#pd_present|pd_copyback,d0
	bra.s		.go4
.wt4:	move.l		#pd_present|pd_writethrough,d0
.go4:	rts

*---------------------------------------------------------------*
*	Mark PMMU tree space as non-cacheable ram		*
*---------------------------------------------------------------*
mask_tree:
*---------------------------------------------------------------*
	movem.l		a0/a1,-(sp)
*---------------------------------------------------------------*
	move.l		tree_root_ptr,a0
	move.l		tree_end_ptr,a1
	jsr		change_2_serialized
*---------------------------------------------------------------*
	movem.l		(sp)+,a0/a1
*---------------------------------------------------------------*
	rts
	
*---------------------------------------------------------------*
detect_faults:
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	bne		.fine
	tst.l		fastram_pages
	ble		.fine
	check		ram_check
	beq		.fine
*---------------------------------------------------------------*
*	Memory check message					*
*---------------------------------------------------------------*
	lea		msg_starinit,a0
	bsr		cconws
*---------------------------------------------------------------*
	moveq		#-1,d0
*---------------------------------------------------------------*
*	Check all available memory				*
*---------------------------------------------------------------*
	clr.w		mcnt
	move.l		alt_start,a0
	move.l		fastram_pages,d5
	divu.l		#(1024*1024)/8192,d5
	bra.s		.ol2s
*---------------------------------------------------------------*
.olp2:	move.l		#(1024*1024)/16,d4
	move.l		a0,a1
*---------------------------------------------------------------*
.lp2:	move.l		(a0),d6
	not.l		(a0)
	not.l		d6
	cmp.l		(a0),d6
	bne.s		.exit
	lea		16(a0),a0
	subq.l		#1,d4
	bne.s		.lp2
	addq.w		#1,mcnt
*---------------------------------------------------------------*
	pushall
*---------------------------------------------------------------*
	cmp.w		#17,mcnt
	bne.s		.n16
	move.w		#1,mcnt
*---------------------------------------------------------------*
	lea		msg_starnl,a0
	bsr		cconws
*---------------------------------------------------------------*
.n16:	lea		msg_star,a0
	bsr		cconws
*---------------------------------------------------------------*
	popall
*---------------------------------------------------------------*
.ol2s:	subq.l		#1,d5
	bpl.s		.olp2
*---------------------------------------------------------------*
	moveq		#0,d0
*---------------------------------------------------------------*
.exit:	push.l		d0
	lea		msg_starfin,a0
	bsr		cconws
	pop.l		d0
.fine:	rts

*---------------------------------------------------------------*
memory_check:
*---------------------------------------------------------------*
	move.l		#block_list,blocklist_ptr
	clr.w		memory_blocks
	clr.l		local_start	
	clr.l		local_end
*---------------------------------------------------------------*
*	Mark all banks for scan					*
*---------------------------------------------------------------*
	lea		bank1,a0
	check		ram_bank2
	beq.s		.only_first
	add.l		#2*(bank_megs*(1024*1024)),a0
	moveq		#2-1,d0
	bra.s		.both_banks
*---------------------------------------------------------------*
.only_first:
*---------------------------------------------------------------*
	add.l		#1*(bank_megs*(1024*1024)),a0
	moveq		#1-1,d0
*---------------------------------------------------------------*
.both_banks:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
.bank_loop:
*---------------------------------------------------------------*
	sub.l		#bank_megs*(1024*1024),a0
	movem.l		d0/a0,-(sp)
	bsr		mark_bank
	movem.l		(sp)+,d0/a0
	dbra		d0,.bank_loop
*---------------------------------------------------------------*

;	lea		bank2.l,a0
;	bsr		mark_bank
;	lea		bank1.l,a0
;	bsr		mark_bank

*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	Scan both banks and log any contiguous blocks		*
*---------------------------------------------------------------*
	lea		bank1,a0
	move.l		#8192,d3
	check		ram_bank2
	beq.s		.only_first2
	move.l		#((bank_megs*2)*(1024*1024))/8192,d4
	bra.s		.both_banks2
*---------------------------------------------------------------*
.only_first2:
*---------------------------------------------------------------*
	move.l		#((bank_megs*1)*(1024*1024))/8192,d4
*---------------------------------------------------------------*
.both_banks2:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Find block start					*
*---------------------------------------------------------------*
.search_start:
*---------------------------------------------------------------*

	move.l		a0,a1
	move.l		a0,d0
	and.l		#$FF000000,d0
	move.w		#$E040,d0
	move.w		sr,d5
	or.w		#$700,sr
	movec		dtt0,d6
	movec		d0,dtt0
	nop
	move.l		(a1),d1
	nop
	movec		d6,dtt0
	move.w		d5,sr

;	move.l		a0,a1
;	move.l		a0,d0
;	and.l		#$FF000000,d0
;	or.l		#$0000C040,d0
;	movec		d0,dtt0
;	nop
;	move.l		(a1),d1

	subq.l		#1,d4
	add.l		d3,a0
	cmp.l		a1,d1
	beq.s		.found_start
	tst.l		d4
	bne.s		.search_start
*---------------------------------------------------------------*
*	Both banks exhausted - stop searching			*
*---------------------------------------------------------------*
	bra		.end_search
*---------------------------------------------------------------*
*	Found start of new bank - now determine length		*
*---------------------------------------------------------------*
.found_start:
*---------------------------------------------------------------*
	move.l		a1,local_start	
*---------------------------------------------------------------*
*	Find block end						*
*---------------------------------------------------------------*
.search_end:
*---------------------------------------------------------------*

	move.l		a0,a1
	move.l		a0,d0
	and.l		#$FF000000,d0
	move.w		#$E040,d0
	move.w		sr,d5
	or.w		#$700,sr
	movec		dtt0,d6
	movec		d0,dtt0
	nop
	move.l		(a1),d1
	nop
	movec		d6,dtt0
	move.w		d5,sr

;	move.l		a0,a1
;	move.l		a0,d0
;	and.l		#$FF000000,d0
;	or.l		#$0000C040,d0
;	movec		d0,dtt0
;	nop
;	move.l		(a1),d1

	subq.l		#1,d4
	add.l		d3,a0
	cmp.l		a1,d1
	bne.s		.found_end
	tst.l		d4
	bne.s		.search_end
*---------------------------------------------------------------*
.found_end:
*---------------------------------------------------------------*
*	Found end of contiguous block - log anything over 1MB	*
*---------------------------------------------------------------*
	move.l		a1,d0
	sub.l		a3,d0
	move.l		d0,local_end
	sub.l		local_start,d0
	cmp.l		#(1024*1024),d0
	bmi.s		.too_small
*---------------------------------------------------------------*
*	Block is 1MB or more - log it in blocklist		*
*---------------------------------------------------------------*
	move.l		blocklist_ptr,a2
	move.l		local_start,(a2)+
	move.l		local_end,(a2)+
	move.l		a2,blocklist_ptr
	addq.w		#1,memory_blocks
	cmp.w		#max_blocks,memory_blocks
	bmi.s		.too_small
	moveq		#0,d4
*---------------------------------------------------------------*
*	Block is too small to be valid memory			*
*---------------------------------------------------------------*
.too_small:
*---------------------------------------------------------------*
*	Look for next block if banks not already exhausted	*
*---------------------------------------------------------------*
	tst.l		d4
	bne		.search_start	
*---------------------------------------------------------------*
.end_search
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
mark_bank:
*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
	move.l		#(8*1024),d3
	move.l		#bank_megs*(1024/8),d4
*---------------------------------------------------------------*
	move.l		d4,d0
	mulu.l		d3,d0
	add.l		d0,a0
*---------------------------------------------------------------*
.loop:	sub.l		d3,a0

;	move.l		a0,a1
;	move.l		a0,d0
;	and.l		#$FF000000,d0
;	or.l		#$0000C040,d0
;	movec		d0,dtt0
;	nop
;	move.l		a1,(a1)

	move.l		a0,a1
	move.l		a0,d0
	and.l		#$FF000000,d0
	move.w		#$E040,d0
	move.w		sr,d5
	or.w		#$700,sr
	movec		dtt0,d6
	movec		d0,dtt0
	nop
	move.l		a1,(a1)
	nop
	movec		d6,dtt0
	move.w		d5,sr

	subq.l		#1,d4
	bne.s		.loop
*---------------------------------------------------------------*
.exit:	rts

*---------------------------------------------------------------*
init_table_pointers:
*---------------------------------------------------------------*
*	Calculate table addresses (8k boundary)			*
*---------------------------------------------------------------*
	move.l		#root_tables,d1
	move.l		#pointer_tables,d2
	move.l		#page_tables,d3
	move.l		#-512,d0
	and.l		d0,d1
	and.l		d0,d2
	and.l		d0,d3
*---------------------------------------------------------------*
*	Relocate tree to [tree_root_ptr] buffer (if required)	*	
*---------------------------------------------------------------*
	sub.l		d1,d2
	add.l		tree_root_ptr,d2
	sub.l		d1,d3
	add.l		tree_root_ptr,d3
	move.l		tree_root_ptr,d1
	move.l		d1,a1
	move.l		d2,a2
	move.l		d3,a3
*---------------------------------------------------------------*
*	Find end of tree					*
*---------------------------------------------------------------*
	move.l		a1,d0
	add.l		#(tree_end-root_tables),d0
	move.l		d0,tree_end_ptr
*---------------------------------------------------------------*
*	Clear tree contents					*
*---------------------------------------------------------------*
	pushall
*---------------------------------------------------------------*
	moveq		#0,d1
	move.l		#(tree_end-root_tables)/4,d0
.all:	move.l		d1,(a1)+
	subq.l		#1,d0
	bne.s		.all
*---------------------------------------------------------------*
	popall
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
mark_8k:
*---------------------------------------------------------------*
	bra		.go
.lp:	move.l		a6,d6
	and.l		#$FE000000,d6
	rol.l		#7,d6
	move.l		(a1,d6.l*4),d2
	bne.s		.rdon
	move.l		a2,d2
	lea		128*4(a2),a2
	and.l		#$FFFFFE00,d2
	or.b		#%11,d2
	move.l		d2,(a1,d6.l*4)
.rdon:	and.l		#$FFFFFE00,d2
	move.l		d2,a0
	move.l		a6,d6
	and.l		#$01FC0000,d6
	lsr.l		#2,d6
	swap		d6
	move.l		(a0,d6.l*4),d3
	bne.s		.pdon
	move.l		a3,d3
	lea		32*4(a3),a3
	and.l		#$FFFFFF80,d3
	or.b		#%11,d3
	move.l		d3,(a0,d6.l*4)
.pdon:	and.l		#$FFFFFF80,d3
	move.l		d3,a0
	move.l		a6,d6
	and.l		#$0003E000,d6
	lsr.l		#13-8,d6
	lsr.l		#8,d6
	move.l		a4,d4
	and.l		#$FFFFE000,d4
	or.l		d0,d4
	move.l		d4,(a0,d6.l*4)
.fdon:	add.l		#(8*1024),a4
	add.l		#(8*1024),a6
.go:	dbra		d7,.lp
	rts

*---------------------------------------------------------------*
*	Convert pages to copyback mode				*
*---------------------------------------------------------------*
change_2_copyback:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*---------------------------------------------------------------*
	movem.l		d0/d1,-(sp)
	move.l		#$FFFFFF9F,d1
	moveq		#pd_copyback,d0
	bsr		modify_descriptors
	movem.l		(sp)+,d0/d1
	rts

*---------------------------------------------------------------*
*	Convert pages to writethrough mode			*
*---------------------------------------------------------------*
change_2_writethrough:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*---------------------------------------------------------------*
	movem.l		d0/d1,-(sp)
	move.l		#$FFFFFF9F,d1
	moveq		#pd_writethrough,d0
	bsr		modify_descriptors
	movem.l		(sp)+,d0/d1
	rts

*---------------------------------------------------------------*
*	Convert pages to serialized mode			*
*---------------------------------------------------------------*
change_2_serialized:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*---------------------------------------------------------------*
	movem.l		d0/d1,-(sp)
	move.l		#$FFFFFF9F,d1
	moveq		#pd_serialized,d0
	bsr		modify_descriptors
	movem.l		(sp)+,d0/d1
	rts

*---------------------------------------------------------------*
*	Convert pages to nonserialized mode			*
*---------------------------------------------------------------*
change_2_nonserialized:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*---------------------------------------------------------------*
	movem.l		d0/d1,-(sp)
	move.l		#$FFFFFF9F,d1
	moveq		#pd_nonserialized,d0
	bsr		modify_descriptors
	movem.l		(sp)+,d0/d1
	rts

*---------------------------------------------------------------*
*	Read PDs from tree into buffer				*
*---------------------------------------------------------------*
read_descriptors:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*	a2.l		PD storage buffer (dst)			*
*---------------------------------------------------------------*
	movem.l		d0-d7/a0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		a0,d6
	move.l		a1,d7
	add.l		#8192-1,d7
	and.l		#-8192,d6
	and.l		#-8192,d7
	sub.l		d6,d7
	move.l		d6,a6
	moveq		#13,d6
	lsr.l		d6,d7
	move.l		a2,a4
	lea		read_pds,a3
	bsr		access_pd_entries
*---------------------------------------------------------------*
	movem.l		(sp)+,d0-d7/a0-a6
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Write PDs from buffer into tree				*
*---------------------------------------------------------------*
write_descriptors:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*	a2.l		PD storage buffer (src)			*
*---------------------------------------------------------------*
	movem.l		d0-d7/a0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		a0,d6
	move.l		a1,d7
	add.l		#8192-1,d7
	and.l		#-8192,d6
	and.l		#-8192,d7
	sub.l		d6,d7
	move.l		d6,a6
	moveq		#13,d6
	lsr.l		d6,d7
	move.l		a2,a4
	lea		write_pds,a3
	bsr		access_pd_entries
*---------------------------------------------------------------*
	movem.l		(sp)+,d0-d7/a0-a6
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Modify PDs with PD field mask & field data		*
*---------------------------------------------------------------*
modify_descriptors:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*	d0.l		PD field data				*
*	d1.l		PD field mask				*
*---------------------------------------------------------------*
	movem.l		d0-d7/a0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		a0,d6
	move.l		a1,d7
	add.l		#8192-1,d7
	and.l		#-8192,d6
	and.l		#-8192,d7
	sub.l		d6,d7
	move.l		d6,a6
	moveq		#13,d6
	lsr.l		d6,d7
	move.l		d1,d5
	lea		modify_pds,a3
	bsr		access_pd_entries
*---------------------------------------------------------------*
	movem.l		(sp)+,d0-d7/a0-a6
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Modify PDs with PD field mask & field data		*
*---------------------------------------------------------------*
xlate_descriptors:
*---------------------------------------------------------------*
*	a0.l		logical page start address		*
*	a1.l		logical page end address		*
*	a2.l		new physical page start address		*
*---------------------------------------------------------------*
	movem.l		d0-d7/a0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		a0,d6
	move.l		a1,d7
	add.l		#8192-1,d7
	and.l		#-8192,d6
	and.l		#-8192,d7
	sub.l		d6,d7
	move.l		d6,a6
	moveq		#13,d6
	lsr.l		d6,d7
	lea		xlate_pds,a3
	bsr		access_pd_entries
*---------------------------------------------------------------*
	movem.l		(sp)+,d0-d7/a0-a6
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
*	Modify the page flags belonging to a group of pages	*
*---------------------------------------------------------------*
access_pd_entries:
*---------------------------------------------------------------*
*	Freeze exceptions					*
*---------------------------------------------------------------*
	push.w		sr
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Freeze cache manager					*
*---------------------------------------------------------------*
	addq.b		#1,tkrd_cacrlock+tkrd_struct
*---------------------------------------------------------------*
*	Flush caches & disable data cache			*
*---------------------------------------------------------------*
	movec		cacr,d1
	move.l		d1,d2
	and.l		#ic4,d1
	nop
	cpusha		bc
 	nop
	movec		d1,cacr
*---------------------------------------------------------------*
*	Restore exceptions					*
*---------------------------------------------------------------*
	pop.w		sr
*---------------------------------------------------------------*
*	Store original cache status for later			*
*---------------------------------------------------------------*
	push.l		d2
*---------------------------------------------------------------*
	move.l		tree_root_ptr,a1
	lea		(8*1024).w,a5
*---------------------------------------------------------------*
*	Look up [pointer table] in [root pointer table]		*
*---------------------------------------------------------------*
.lp:	move.l		a6,d6
	bfextu		d6{(31-31):07},d1
	xtree		<(a1,d1.l*4)>,d2
*---------------------------------------------------------------*
*	Validate entry						*
*---------------------------------------------------------------*
	moveq		#%11,d3
	and.l		d2,d3
	beq.s		.err
*---------------------------------------------------------------*
*	Turn entry into [pointer table] address			*
*---------------------------------------------------------------*
	bfclr		d2{(31-08):09}
	move.l		d2,a0
*---------------------------------------------------------------*
*	Look up [page table] in [pointer table]			*
*---------------------------------------------------------------*
	move.l		a6,d6
	bfextu		d6{(31-24):07},d1
	xtree		<(a0,d1.l*4)>,d2
*---------------------------------------------------------------*
*	Validate entry						*
*---------------------------------------------------------------*
	moveq		#%11,d3
	and.l		d2,d3
	beq.s		.err
*---------------------------------------------------------------*
*	Turn entry into [page table] address			*
*---------------------------------------------------------------*
	bfclr		d2{(31-06):07}
	move.l		d2,a0
*---------------------------------------------------------------*
*	Look up [page descriptor] in [page table]		*
*---------------------------------------------------------------*
	move.l		a6,d6
	bfextu		d6{(31-17):05},d1
*---------------------------------------------------------------*
*	Call function-specific routine				*
*---------------------------------------------------------------*
	jsr		(a3)
*---------------------------------------------------------------*
*	Advance PD record buffer (if used)			*
*---------------------------------------------------------------*
.err:	addq.l		#4,a4
*---------------------------------------------------------------*
*	Advance logical and physical address pointers		*
*---------------------------------------------------------------*
	add.l		a5,a6
	add.l		a5,a2
*---------------------------------------------------------------*
	subq.l		#1,d7
	bne		.lp
*---------------------------------------------------------------*
*	Retrieve original cache configuration			*
*---------------------------------------------------------------*
.skip:	pop.l		d1
*---------------------------------------------------------------*
*	Freeze exceptions					*
*---------------------------------------------------------------*
	push.w		sr
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Unlock cache manager					*
*---------------------------------------------------------------*
	subq.b		#1,tkrd_cacrlock+tkrd_struct
*---------------------------------------------------------------*
*	Flush CPU caches & restore cache configuration		*
*---------------------------------------------------------------*
	nop
	cpusha		bc
 	nop
	movec		d1,cacr
*---------------------------------------------------------------*
*	Restore exceptions					*
*---------------------------------------------------------------*
	pop.w		sr
*---------------------------------------------------------------*
	rts

*---------------------------------------------------------------*
read_pds:
*---------------------------------------------------------------*
*	Fetch page descriptor from tree				*
*---------------------------------------------------------------*
	xtree		<(a0,d1.l*4)>,d2
*---------------------------------------------------------------*
*	Store [page descriptor] in record buffer		*
*---------------------------------------------------------------*
	move.l		d2,(a4)
	rts

*---------------------------------------------------------------*
write_pds:
*---------------------------------------------------------------*
*	Fetch page descriptor from buffer			*
*---------------------------------------------------------------*
	move.l		(a4),d2
*---------------------------------------------------------------*
*	Store [page descriptor] in tree				*
*---------------------------------------------------------------*
	xtree		d2,<(a0,d1.l*4)>
	rts

*---------------------------------------------------------------*
modify_pds:
*---------------------------------------------------------------*
*	Modify [page descriptor] & return it to [page table]	*
*---------------------------------------------------------------*
	xtree		<(a0,d1.l*4)>,d2
	and.l		d5,d2
	or.l		d0,d2
	xtree		d2,<(a0,d1.l*4)>
	rts

*---------------------------------------------------------------*
xlate_pds:
*---------------------------------------------------------------*
*	Read & validate [page descriptor] from [page table]	*
*---------------------------------------------------------------*
	xtree		<(a0,d1.l*4)>,d2
	moveq		#%11,d3
	and.l		d2,d3
	beq.s		.err
*---------------------------------------------------------------*
*	Xlate [page descriptor] & return it to [page table]	*
*---------------------------------------------------------------*
	move.l		a2,d5
	and.l		#$FFFE0000,d2
	and.l		#$FFFE0000,d5
	or.l		d5,d2
	xtree		d2,<(a0,d1.l*4)>
.err:	rts


*---------------------------------------------------------------*
	cnop		0,4
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_copyback_flush:
*---------------------------------------------------------------*
	ds.l		1
*---------------------------------------------------------------*
copyback_flush:
*---------------------------------------------------------------*
	cpusha		dc
;	push.l		a0
;	push.l		a1
;	move.l		last_flush,a0
;	move.l		$44e.w,a1
;	cmp.l		a1,a0
;	bgt.s		.cont
;	move.l		$42e.w,a0
;.cont:	lea		-8192(a0),a0
;	cpushp		dc,(a0)
;	move.l		a0,last_flush
;	pop.l		a1
;	pop.l		a0
	jmp		([old_copyback_flush.l])

*---------------------------------------------------------------*
	cnop		0,4
*---------------------------------------------------------------*
	dc.l		'XBRA'
	dc.l		ID
*---------------------------------------------------------------*
old_nova_blanker:
*---------------------------------------------------------------*
	ds.l		1
*---------------------------------------------------------------*
nova_blanker:
*---------------------------------------------------------------*
	move.w		$FFFF82A2.w,$FFFF82A4.w
	move.w		$FFFF82A2.w,$FFFF82A6.w
	move.w		#-1,$FFFF82A8.w
	jmp		([old_nova_blanker.l])
	
*---------------------------------------------------------------*
*	Patch TOS to remove MOVEM clear sequences		*
*---------------------------------------------------------------*
patch_fastrom:
*---------------------------------------------------------------*
*	Skip patches if Magic is resident			*
*---------------------------------------------------------------*
	tst.b		magic_resident
	bne		.nmp
*---------------------------------------------------------------*
*	Skip patches if FastROM is resident			*
*---------------------------------------------------------------*
	tst.b		pmmu_resident
	bne		.nmp
*---------------------------------------------------------------*
*	Skip patches if FastROM not available			*
*---------------------------------------------------------------*
	tst.b		fastrom_installed
	beq		.nmp
*---------------------------------------------------------------*
*	Remove MOVEM.L based blockfill routines			*
*---------------------------------------------------------------*
	push.w		sr
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Upload code segments to ROM				*
*---------------------------------------------------------------*
	lea		rom_block_start,a0
	lea		rom_block_end,a1
	lea		rom_block_base,a2
	bsr		copy_code
*---------------------------------------------------------------*
	lea		clrcode1_start,a0
	lea		clrcode1_end,a1
	lea		$E0016A,a2
	bsr		copy_code
	lea		clrcode2_start,a0
	lea		clrcode2_end,a1
	lea		$E01464,a2
	bsr		copy_code
	lea		clrcode2_start,a0
	lea		clrcode2_end,a1
	lea		$E10572,a2
	bsr		copy_code
*---------------------------------------------------------------*
*	Patch incorrect 68020 cookie to 68040 at boot time	*
*---------------------------------------------------------------*
	lea		cookiecode_nologo_start,a0
	lea		cookiecode_nologo_end,a1
	check		rom_logo
	beq.s		.nlog2
	lea		cookiecode_logo_start,a0
	lea		cookiecode_logo_end,a1
.nlog2:	lea		$E003D4,a2
	bsr		copy_code
*---------------------------------------------------------------*
*	Init caches & PMMU at boot time				*
*---------------------------------------------------------------*
	lea		cacrcode1_start,a0
	lea		cacrcode1_end,a1
	lea		$E0005A,a2
	bsr		copy_code
*---------------------------------------------------------------*
*	Init Nemesis at boot time				*
*---------------------------------------------------------------*
	lea		nemcode_initoff_start,a0
	lea		nemcode_initoff_end,a1
	check		sys_nemesis
	beq.s		.non1
	lea		nemcode_initon_start,a0
	lea		nemcode_initon_end,a1
.non1:	lea		$E0006A,a2
	bsr		copy_code
*---------------------------------------------------------------*
	lea		nemcode2_start,a0
	lea		nemcode2_end,a1
	lea		$E0008E,a2
	bsr		copy_code
*---------------------------------------------------------------*
	check		sys_nemesis
	beq.s		.non2
	lea		nemcode_on_start,a0
	lea		nemcode_on_end,a1
	lea		$E0270E,a2
	bsr		copy_code	
*---------------------------------------------------------------*
*	Patch 68040 cache access (data cache flush)		*
*---------------------------------------------------------------*	
.non2:

	rept		0
	
	lea		fdc_12_code_start,a0
	lea		fdc_12_code_end,a1
	lea		$E0194A,a2
	bsr		copy_code
	lea		$E11BCE,a2
	bsr		copy_code
	lea		$E11C56,a2
	bsr		copy_code
	lea		$E11D52,a2
	bsr		copy_code
	lea		$E125B8,a2
	bsr		copy_code
	lea		$E1292E,a2
	bsr		copy_code
	lea		$E1297A,a2
	bsr		copy_code
	lea		$E129DE,a2
	bsr		copy_code
	lea		$E12BF6,a2
	bsr		copy_code
	lea		$E12C76,a2
	bsr		copy_code
	lea		$E12CEE,a2
	bsr		copy_code
	lea		$E12D6C,a2
	bsr		copy_code
	lea		$E12DD4,a2
	bsr		copy_code
	lea		$E12E96,a2
	bsr		copy_code
	lea		$E17B4C,a2
	bsr		copy_code
	lea		$E17BE4,a2
	bsr		copy_code

	endr

*---------------------------------------------------------------*
*	Patch 68040 cache access (disable both caches)		*
*---------------------------------------------------------------*
	lea		fbcdbcd0_10_code_start,a0
	lea		fbcdbcd0_10_code_end,a1
	lea		$E00756,a2
	bsr		copy_code
*---------------------------------------------------------------*
*	Patch 68040 cache access (flush both caches)		*
*---------------------------------------------------------------*
	lea		fbc_12_code_start,a0
	lea		fbc_12_code_end,a1
	lea		$E018D0,a2
	bsr		copy_code
*---------------------------------------------------------------*
	lea		fbc_14_code_start,a0
	lea		fbc_14_code_end,a1
	lea		$E00860,a2
	bsr		copy_code

*---------------------------------------------------------------*
*	Patch 68040 cache access (flush instruction cache)	*
*---------------------------------------------------------------*
	lea		fic_12_code_start,a0
	lea		fic_12_code_end,a1
	lea		$E0990C,a2
	bsr		copy_code
*---------------------------------------------------------------*
	lea		rc1_code_start,a0
	lea		rc1_code_end,a1
	lea		$E250C8,a2
;	bsr		copy_code
*---------------------------------------------------------------*
	lea		rc2_code_start,a0
	lea		rc2_code_end,a1
	lea		$E444F6,a2
;	bsr		copy_code
*---------------------------------------------------------------*
	lea		rc3_code_start,a0
	lea		rc3_code_end,a1
	lea		$E44526,a2
;	bsr		copy_code

*---------------------------------------------------------------*
*	Fix CTRL-ALT-SHIFT-DEL coldstart keys			*
*---------------------------------------------------------------*
	lea		rescode_start,a0
	lea		rescode_end,a1
	lea		$E0398C,a2
	bsr		copy_code
*---------------------------------------------------------------*
*	Patch line-aligned malloc				*
*---------------------------------------------------------------*

	ifd		align_malloc

	lea		malloc_start,a0
	lea		malloc_end,a1
	lea		$E1C6B8,a2
	bsr		copy_code

	lea		malloc_start,a0
	lea		malloc_end,a1
	lea		$E1C838,a2
	bsr		copy_code

	lea		malloc2_start,a0
	lea		malloc2_end,a1
	lea		$E1C660,a2
	bsr		copy_code
	
	endc
	
*---------------------------------------------------------------*
*	Patch ROM Pexec						*
*---------------------------------------------------------------*
	check		cpu_pexec
	beq.s		.no_lock
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
	move.l		$E50186,(old_rom_pexec+rom_block_base-rom_block_start)
	move.l		#(rom_pexec+rom_block_base-rom_block_start),$E50186
*---------------------------------------------------------------*
.no_lock:
*---------------------------------------------------------------*
*	Replace boot logo					*
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
	check		rom_logo
	beq.s		.nlogo
	add.l		#logo,logo
	move.l		#logo,$E00662
*---------------------------------------------------------------*
.nlogo:	move.w		#$4e71,$E00038
	move.w		#$6000,$E006AC
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
	pop.w		sr
*---------------------------------------------------------------*
*	Recalculate ROM CRC					*
*---------------------------------------------------------------*
	push.w		#1
	push.l		#(512*1024)-2
	push.l		#$E00000
	jsr		$E01508
	lea		10(sp),sp
	move.w		d0,$E00000+(512*1024)-2	
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
*	Write-protect ROM area					*
*---------------------------------------------------------------*
	check		rom_wp
	beq.s		.nmp
*---------------------------------------------------------------*
	lea		$00E00000,a0
	lea		$00E80000,a1
	moveq		#pd_writeprotect,d0
	move.l		#$FFFFFFFB,d1
	jsr		modify_descriptors
.nmp:	rts

*---------------------------------------------------------------*
copy_code:
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
	movem.l		a0/a1,-(sp)
	move.l		a1,d0
	sub.l		a0,d0
	subq.l		#1,d0
.copy:	move.b		(a0)+,(a2)+
	dbra		d0,.copy
	movem.l		(sp)+,a0/a1
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
	rts
	
*---------------------------------------------------------------*
clrcode1_start:
*---------------------------------------------------------------*
	movem.l		d0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		d5,d0
	sub.l		a4,d0		; total size
	ble.s		.done
*---------------------------------------------------------------*
	moveq		#0,d2
	lsr.l		#2,d0
	moveq		#4-1,d1
	and.l		d0,d1
	lsr.l		#2,d0
	neg.l		d1
	jmp		.jmp(pc,d1.l*2)
.loop:	move.l		d2,(a4)+
	move.l		d2,(a4)+
	move.l		d2,(a4)+
	move.l		d2,(a4)+
.jmp:	subq.l		#1,d0
	bpl.s		.loop
*---------------------------------------------------------------*
.done:	movem.l		(sp)+,d0-a6
*---------------------------------------------------------------*
	bra.s		clrcode1_end
*---------------------------------------------------------------*
	rept		27/2
	nop
	endr
*---------------------------------------------------------------*
clrcode1_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
clrcode2_start:
*---------------------------------------------------------------*
	movem.l		d0-a6,-(sp)
*---------------------------------------------------------------*
	move.l		(15*4)+4(sp),a4
	move.l		(15*4)+8(sp),d5
*---------------------------------------------------------------*
	move.l		d5,d0
	sub.l		a4,d0		; total size
	ble.s		.done
	move.l		a4,a0
	moveq		#0,d2
	jsr		(turbo_memset+rom_block_base-rom_block_start)
*---------------------------------------------------------------*
.done:	movem.l		(sp)+,d0-a6
*---------------------------------------------------------------*
	rts
*---------------------------------------------------------------*
clrcode2_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
cookiecode_nologo_start:
*---------------------------------------------------------------*
	rept		(13-8)/2
	nop
	endr
	nop
	pflusha
	nop

	rept		12
	nop
	endr

;	move.l		$432.l,d1
;	add.l		#8192-1,d1
;	and.w		#-8192,d1
;	move.l		d1,$4fa.w
;	move.l		d1,$432.w

	moveq		#40,d1
*---------------------------------------------------------------*
cookiecode_nologo_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
cookiecode_logo_start:
*---------------------------------------------------------------*
	rept		(13-8)/2
	nop
	endr
	nop
	pflusha
	nop
	move.l		#prog_end,d1
	add.l		#8192-1,d1
	and.w		#-8192,d1
	move.l		d1,$4fa.w
	move.l		d1,$432.w
	moveq		#40,d1
*---------------------------------------------------------------*
cookiecode_logo_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
cacrcode1_start:
*---------------------------------------------------------------*
	lea		.rts(pc),a6
	jmp		(init_mmu+rom_block_base-rom_block_start)
.rts
*---------------------------------------------------------------*
cacrcode1_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
nemcode_initon_start:
*---------------------------------------------------------------*
	rept		(24-10)/2
	nop
	endr
	lea		.rts(pc),a0
	jmp		(nemcode_on+rom_block_base-rom_block_start)
.rts:
*---------------------------------------------------------------*
nemcode_initon_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
nemcode_initoff_start:
*---------------------------------------------------------------*
	rept		(24-10)/2
	nop
	endr
	lea		.rts(pc),a0
	jmp		(nemcode_off+rom_block_base-rom_block_start)
.rts:
*---------------------------------------------------------------*
nemcode_initoff_end:
*---------------------------------------------------------------*
	
*---------------------------------------------------------------*
nemcode2_start:
*---------------------------------------------------------------*
	nop
	nop
	nop
*---------------------------------------------------------------*
nemcode2_end:
*---------------------------------------------------------------*

	ifd		align_malloc

*---------------------------------------------------------------*
malloc_start:
*---------------------------------------------------------------*
	addq.l		#8,d7
	addq.l		#8-1,d7
	and.w		#-16,d7
	nop
*---------------------------------------------------------------*
malloc_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
malloc2_start:
*---------------------------------------------------------------*
	jmp		round_malloc
*---------------------------------------------------------------*
malloc2_end:
*---------------------------------------------------------------*

	endc

*---------------------------------------------------------------*
nemcode_on_start:
*---------------------------------------------------------------*
	move.b		#$03,$fffffc00.w
	move.b		#$d6,$fffffc00.w
*---------------------------------------------------------------*
nemcode_on_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
rescode_start:
*---------------------------------------------------------------*
	jmp		(coldstart+rom_block_base-rom_block_start)
*---------------------------------------------------------------*
rescode_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
fbcdbcd0_10_code_start:
*---------------------------------------------------------------*

;	bra.s		fbcdbcd0_10_code_end
;	rept		8/2
;	nop
;	endr

;	moveq		#0,d0
;	movec		d0,cacr

	nop
	nop
	nop

	cpusha		bc
	nop

*---------------------------------------------------------------*
fbcdbcd0_10_code_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
fbc_14_code_start:
*---------------------------------------------------------------*
fbc_12_code_start:
*---------------------------------------------------------------*
	nop
	cpusha		bc
	nop
	nop
	nop
	nop	
*---------------------------------------------------------------*
fbc_12_code_end:
*---------------------------------------------------------------*
	nop
*---------------------------------------------------------------*
fbc_14_code_end:
*---------------------------------------------------------------*
	
*---------------------------------------------------------------*
fdc_12_code_start:
*---------------------------------------------------------------*

	bra.s		fdc_12_code_end
	rept		10/2
	nop
	endr

;	nop
;	nop
;	nop
;	nop
;	cpusha		dc
;	nop

*---------------------------------------------------------------*
fdc_12_code_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
fic_12_code_start:
*---------------------------------------------------------------*
	nop
	cinva		ic
	nop
	nop
	nop
	nop
*---------------------------------------------------------------*
fic_12_code_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
rc1_code_start:
*---------------------------------------------------------------*
	move.l		#$3111,d0	
	rts
*---------------------------------------------------------------*
rc1_code_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
rc2_code_start:
*---------------------------------------------------------------*
	move.l		#$3111,d0
	rts
*---------------------------------------------------------------*
rc2_code_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
rc3_code_start:
*---------------------------------------------------------------*

	bra.s		rc3_code_end
	rept		10/2
	nop
	endr

;	rept		12/2
;	nop
;	endr

*---------------------------------------------------------------*
rc3_code_end:
*---------------------------------------------------------------*

	ifd		align_malloc

*---------------------------------------------------------------*
round_malloc:
*---------------------------------------------------------------*
	tst.l		d0
	ble.s		.ok
	sub.l		#16,d0
	bpl.s		.ok
	moveq		#0,d0
.ok:	movem.l		(sp)+,d6-d7/a5
	unlk		a6
	rts
	
	endc

*---------------------------------------------------------------*
*	Text output						*
*---------------------------------------------------------------*
cconws:
*---------------------------------------------------------------*
	pea		(a0)
	push.w		#9
	trap		#1
	addq.l		#6,sp
	rts

*---------------------------------------------------------------*
*	Routines uploaded to ROM				*
*---------------------------------------------------------------*

*---------------------------------------------------------------*
rom_block_start:
*---------------------------------------------------------------*

*-------------------------------------------------------*
turbo_memset:
*-------------------------------------------------------*
	move.l		d2,d1
	swap		d2
	or.l		d2,d1
	move.l		d1,d2
	rol.l		#8,d1
	or.l		d1,d2
*-------------------------------------------------------*
.long_fill:
*-------------------------------------------------------*
	moveq		#16-1,d1
	add.l		a0,d1
	and.w		#-16,d1
	sub.l		a0,d1
	sub.l		d1,d0	
	bpl.s		.no_cliplong
	add.l		d0,d1
	moveq		#0,d0
.no_cliplong:
*-------------------------------------------------------*
*	0->15 byte preamble				*
*-------------------------------------------------------*
	lsr.l		d1
	bcc.s		.prelong_0
	move.b		d2,(a0)+
.prelong_0:
	lsr.l		d1
	bcc.s		.prelong_start
	move.w		d2,(a0)+
	bra.s		.prelong_start
.prelong_loop:
	move.l		d2,(a0)+
.prelong_start:
	dbra		d1,.prelong_loop
*-------------------------------------------------------*
*	Aligned long-transfer				*
*-------------------------------------------------------*
	moveq		#-16,d1
	and.l		d0,d1
	beq.s		.long_done	
	sub.l		d1,d0
	lsr.l		#4,d1
.long_transfer:
	move.l		d2,(a0)+
	move.l		d2,(a0)+
	move.l		d2,(a0)+
	move.l		d2,(a0)+
	subq.l		#1,d1
	bne.s		.long_transfer
.long_done:
*-------------------------------------------------------*
*	0->15 byte postamble				*
*-------------------------------------------------------*
	moveq		#-4,d1
	and.l		d0,d1
	sub.l		d1,d0
	lsr.l		#2,d1
	bra.s		.postlong_start
.postlong_loop:
	move.l		d2,(a0)+
.postlong_start:
	dbra		d1,.postlong_loop
.postlong_done:
	bclr		#1,d0
	beq.s		.postlong_1
	move.w		d2,(a0)+
.postlong_1:
	bclr		#0,d0
	beq.s		.postlong_0
	move.b		d2,(a0)+
.postlong_0:
	rts

*---------------------------------------------------------------*
init_mmu:
*---------------------------------------------------------------*
*	Lock out interrupts					*
*---------------------------------------------------------------*
	move.w		sr,d7
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Record xTTx register states				*
*---------------------------------------------------------------*
	movec		dtt0,d1
*---------------------------------------------------------------*
*	Disable caches						*
*---------------------------------------------------------------*
	moveq		#0,d0
	movec		d0,cacr
	nop
	cinva		bc
	nop
*---------------------------------------------------------------*
*	Transparently translate all RAM				*
*---------------------------------------------------------------*
	move.l		#$00FFE040,d0
	movec		d0,dtt0	
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	Flag for 'no tree present'				*
*---------------------------------------------------------------*
	moveq		#0,d6
*---------------------------------------------------------------*
*	PMMU is invalid if SRP is not in STRAM			*
*---------------------------------------------------------------*
	movec		srp,d0
	tst.l		d0
	beq.s		.no_tree
	cmp.l		#$05000000,d0
	bhs.s		.no_tree
	cmp.l		#$00E00000,d0
	blo.s		.cont
	cmp.l		#$01000000,d0
	blo.s		.no_tree
.cont:	move.l		d0,a0
*---------------------------------------------------------------*
*	Ensure PMMU tree is actually real data			*
*---------------------------------------------------------------*
	cmp.l		#'TREE',mmu_tag(a0)
	bne		.no_tree
	cmp.l		#'KEEP',mmu_id(a0)
	bne.s		.no_tree
*---------------------------------------------------------------*
*	Flag for 'tree present'					*
*---------------------------------------------------------------*
	moveq		#1,d6
*---------------------------------------------------------------*
.no_tree:
*---------------------------------------------------------------*
*	Restore ram state & interrupts				*
*---------------------------------------------------------------*
	nop
	movec		d1,dtt0
	nop
	pflusha
	nop
*---------------------------------------------------------------*
*	If tree is valid, leave MMU alone			*
*---------------------------------------------------------------*
	tst.l		d6
	beq.s		.none
*---------------------------------------------------------------*
*	Restore PMMU						*
*---------------------------------------------------------------*
	move.l		a0,d0
	movec		d0,srp
	movec		d0,urp
	move.l		#$0000C000,d0
	movec		d0,tc
	moveq		#0,d0
	movec		d0,itt0
	movec		d0,itt1
	movec		d0,dtt0
	movec		d0,dtt1
	nop
	pflusha
	nop
	bra.s		.done
*---------------------------------------------------------------*
*	PMMU tree is not valid					*
*---------------------------------------------------------------*
.none:	move.l		#$00FFE000,d0
	movec		d0,itt0
	move.l		#$00FFE040,d0
	movec		d0,dtt0
	moveq		#0,d0
	movec		d0,itt1
	movec		d0,dtt1
	moveq		#0,d0
	movec		d0,tc
	movec		d0,srp
	movec		d0,urp
	nop
	pflusha
	nop
*---------------------------------------------------------------*
.done:	move.l		#ic4,d0
	movec		d0,cacr
*---------------------------------------------------------------*
	move.w		d7,sr
*---------------------------------------------------------------*
	jmp		(a6)

*---------------------------------------------------------------*
nemcode_on:
*---------------------------------------------------------------*
	bclr		#2,$ffff8007.w		; half-clock Blitter
;	bclr		#3,$ffff8007.w		; disable Blitter
	move.b		#$03,$fffffc04.w
	move.b		#$95,$fffffc04.w	; use Nemesis clock #1
	bset		#0,$ffff8007.w		; full-clock CPU
	move.b		#$03,$fffffc00.w
	move.b		#$d6,$fffffc00.w	; set Nemesis on/off
	bra.s		reset_hardware

*---------------------------------------------------------------*
nemcode_off:
*---------------------------------------------------------------*
	bset		#2,$ffff8007.w		; half-clock Blitter
;	bclr		#3,$ffff8007.w		; disable Blitter
	move.b		#$03,$fffffc04.w
	move.b		#$95,$fffffc04.w	; use Nemesis clock #1
	bset		#0,$ffff8007.w		; full-clock CPU
	move.b		#$03,$fffffc00.w
	move.b		#$96,$fffffc00.w	; set Nemesis on/off

*---------------------------------------------------------------*
reset_hardware:
*---------------------------------------------------------------*
	bclr		#5,$FFFF8266.w
	bclr		#6,$FFFF8266.w
	bclr		#0,$FFFF820A.w
	jmp		(a0)

*---------------------------------------------------------------*
coldstart:
*---------------------------------------------------------------*
	or.w		#$700,sr
*---------------------------------------------------------------*
*	Empty palette						*
*---------------------------------------------------------------*
	moveq		#0,d1
	lea		$FFFF9800.w,a0
	move.w		#256-1,d0
.clr1:	move.l		d1,(a0)+
	dbra		d0,.clr1
	lea		$FFFF8240.w,a0
	moveq		#(16/2)-1,d0
.clr2:	move.l		d1,(a0)+
	dbra		d0,.clr2
*---------------------------------------------------------------*
*	Reset hardware						*
*---------------------------------------------------------------*
;	move.b		#$03,$fffffc04.w
;	move.b		#$95,$fffffc04.w	; use Nemesis clock #1
;	bset		#0,$ffff8007.w		; full-clock CPU
;	move.b		#$03,$fffffc00.w
;	move.b		#$96,$fffffc00.w	; set Nemesis on/off
;	nop
*---------------------------------------------------------------*
*	Caches off						*
*---------------------------------------------------------------*
	moveq		#0,d0
	movec		d0,cacr
	nop
*---------------------------------------------------------------*
*	Relocate PMMU reset code to safe memory			*
*---------------------------------------------------------------*
	lea		.local_start,a0
	move.l		#.local_end,d0
	lea		coldreset_base,a1
	sub.l		a0,d0
	addq.l		#4-1,d0
	lsr.l		#2,d0
.copy:	move.l		(a0)+,(a1)+
	subq.l		#1,d0
	bne.s		.copy
*---------------------------------------------------------------*
*	Execute PMMU reset sequence and reboot			*
*---------------------------------------------------------------*
	jmp		coldreset_base
*---------------------------------------------------------------*

*---------------------------------------------------------------*
.local_start:
*---------------------------------------------------------------*
	nop
	cinva		bc
	nop
*---------------------------------------------------------------*
	lea		8.w,a0
	move.l		#coldreset_base,d0
	sub.l		a0,d0
	lsr.l		#2,d0
	clr.l		d1
.clr:	move.l		d1,(a0)+
	subq.l		#1,d0
	bne.s		.clr
*---------------------------------------------------------------*
	nop
	pflusha
	nop
*---------------------------------------------------------------*
	move.l		#$00FFE000,d0
	movec		d0,itt0
	move.l		#$00FFE040,d0
	movec		d0,dtt0
	moveq		#0,d0
	movec		d0,itt1
	movec		d0,dtt1
	nop
	moveq		#0,d0
	movec		d0,tc
	nop
	movec		d0,srp
	movec		d0,urp
	nop
*---------------------------------------------------------------*
	reset
*---------------------------------------------------------------*
	jmp		([$4.w])
*---------------------------------------------------------------*
.local_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Disable instruction cache before code relocation	*
*---------------------------------------------------------------*
rom_pexec:
*---------------------------------------------------------------*
*	Disable instruction cache				*
*---------------------------------------------------------------*
	movec		cacr,d0
	bclr		#ic4_bit,d0
	beq.s		.local_smc
*---------------------------------------------------------------*
*	If cache was previously enabled, then flush it		*
*---------------------------------------------------------------*
	movec		d0,cacr
	nop
	cinva		ic
	nop
*---------------------------------------------------------------*
.local_smc:
*---------------------------------------------------------------*
	jmp		$ABCDABCD

old_rom_pexec	=	.local_smc+2

*---------------------------------------------------------------*
rom_block_end:
*---------------------------------------------------------------*

	include		vdipatch.s

*---------------------------------------------------------------*
		data
*---------------------------------------------------------------*

msg_installed:		dc.b	'Afterburner ToolKit driver - now resident',13,10
			dc.b	0

msg_searching:		dc.b	'Searching SIMM banks for resident pages..',13,10
			dc.b	0

msg_transferring:	dc.b	'Transferring 512k ROM image to FastRAM...',13,10
			dc.b	0


msg_already:		dc.b	27,'E'
			dc.b	'Warning! The driver is already installed!',13,10
			dc.b	0

msg_nofastram:		dc.b	27,'E'
			dc.b	'Error! This driver needs FastRAM to work!',13,10
			dc.b	0

msg_starinit:		dc.b	13,10
			dc.b	'Local 32-bit RAM: '
			dc.b	0

msg_starfin:		dc.b	13,10,13,10
			dc.b	0

msg_star:		dc.b	''
			dc.b	0
			even

msg_starnl:		dc.b	13,10
			dc.b	'                  '
			dc.b	0
			even

*---------------------------------------------------------------*
*	PMMU structure						*
*---------------------------------------------------------------*

*---------------------------------------------------------------*
pmmu_struct:		
*---------------------------------------------------------------*
tree_root_ptr:		dc.l	0
tree_end_ptr:		dc.l	0
*---------------------------------------------------------------*
			dc.l	read_descriptors
			dc.l	write_descriptors
*---------------------------------------------------------------*
			dc.l	change_2_writethrough
			dc.l	change_2_copyback
			dc.l	change_2_serialized
			dc.l	change_2_nonserialized
*---------------------------------------------------------------*
			dc.l	modify_descriptors
			dc.l	xlate_descriptors
*---------------------------------------------------------------*

*---------------------------------------------------------------*
logo_start:
*---------------------------------------------------------------*
logo:			incbin	'bootlogo.mfd'
*---------------------------------------------------------------*
logo_end:		even
*---------------------------------------------------------------*

*---------------------------------------------------------------*
			bss
*---------------------------------------------------------------*
bss_start:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Resident variables					*
*---------------------------------------------------------------*

last_flush:		ds.l	1
current_scrc:		ds.l	1

*---------------------------------------------------------------*
*	TKRD structure						*
*---------------------------------------------------------------*

tkrd_struct:		ds.b	tkrd_slen

*---------------------------------------------------------------*
*	GEMRAM buffer						*
*---------------------------------------------------------------*

usrs_space:		ds.l	256

*---------------------------------------------------------------*
new_jar:		ds.l	2*jar_size
*---------------------------------------------------------------*
endcookies:		ds.l	1
maxcookies:		ds.l	1
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	1 page worth of rounding-down space			*
*---------------------------------------------------------------*
tree_start:		ds.b	8192
*---------------------------------------------------------------*
*	Space for root pointers					*
*---------------------------------------------------------------*
root_tables:		ds.l	128
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Space for pointer tables				*
*---------------------------------------------------------------*
pointer_tables:		ds.l	(pmmu_mapsize/32)*128
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Space for page tables					*
*---------------------------------------------------------------*
page_tables:		ds.l	(pmmu_mapsize/32)*128*32
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	1 page worth of rounding-up space			*
*---------------------------------------------------------------*
tree_end:		ds.b	8192
*---------------------------------------------------------------*

*---------------------------------------------------------------*
prog_end:
*---------------------------------------------------------------*

*---------------------------------------------------------------*
*	Temporary variables					*
*---------------------------------------------------------------*

basepage:		ds.l	1
stay_resident:		ds.l	1
os_ssp_stack:		ds.l	1
os_usp_stack:		ds.l	1
pterm_length:		ds.l	1
magic_resident:		ds.l	1
pmmu_resident:		ds.l	1
fastrom_installed:	ds.l	1
fastram_pageflags:	ds.l	1
fastram_pages:		ds.l	1
sys_start:		ds.l	1
local_start:		ds.l	1
local_end:		ds.l	1
rom_size:		ds.l	1
shadow_base:		ds.l	1
memory_blocks:		ds.l	1
rom_start:		ds.l	1
alt_start:		ds.l	1
alt_end:		ds.l	1
mcnt:			ds.l	1
old_fastram_pages:	ds.l	1

*---------------------------------------------------------------*
*	FastRAM block list					*
*---------------------------------------------------------------*

blocklist_ptr:		ds.l	1
block_list:		ds.l	2*max_blocks

*---------------------------------------------------------------*
			ds.b	user_stacksize
usp_stack:		ds.l	1
*---------------------------------------------------------------*
			ds.b	super_stacksize
ssp_stack:		ds.l	1
*---------------------------------------------------------------*

*---------------------------------------------------------------*
bss_end:
*---------------------------------------------------------------*
